// Logic for the step-by-step building of a repertoire

export function initBuildFlow({ containerSelector, board, game, generateRepertoire, updateMainUI, toDests, toggleBoardInteraction, toColor, getLastMove }) {
  function disableExplorerTab() {
    const explorerTabLink = document.querySelector('.tab-link[data-tab="explorer-tab"]');
    if (explorerTabLink) {
      explorerTabLink.classList.add('disabled-tab');
      explorerTabLink.style.opacity = '0.5';
      explorerTabLink.style.cursor = 'not-allowed';
      if (!explorerTabLink._originalClickHandler) {
        const originalHandler = explorerTabLink.onclick;
        explorerTabLink._originalClickHandler = originalHandler;
        explorerTabLink.onclick = (e) => {
          e.preventDefault();
          e.stopPropagation();
          return false;
        };
      }
    }
  }
  function enableExplorerTab() {
    const explorerTabLink = document.querySelector('.tab-link[data-tab="explorer-tab"]');
    if (explorerTabLink) {
      explorerTabLink.classList.remove('disabled-tab');
      explorerTabLink.style.opacity = '';
      explorerTabLink.style.cursor = '';
      if (explorerTabLink._originalClickHandler) {
        explorerTabLink.onclick = explorerTabLink._originalClickHandler;
        explorerTabLink._originalClickHandler = null;
      } else {
        explorerTabLink.onclick = null;
      }
    }
  }
  let mainBoardOrientation = null;
  try {
    const boardWrapper = document.querySelector('.board-wrapper');
    if (boardWrapper && boardWrapper._chessground) {
      mainBoardOrientation = boardWrapper._chessground.state.orientation;
    }
  } catch (e) {
  }
  const state = {
    currentStep: 0,
    previousStep: null,
    fen: '',
    openingName: '',
    savedPgn: null,
    initialBoardState: null,
    startingBoardOrientation: null,
    color: 'white',
    mode: 'light',
    temperature: 'balanced',
    _stateHistory: []
  };
  function trackStateChange(reason) {
    const currentState = {
      ...state,
      reason,
      timestamp: new Date().toISOString(),
      boardOrientation: board.state.orientation,
      gameFen: game.fen(),
      gamePgn: game.pgn()
    };
    delete currentState._stateHistory;
    state._stateHistory.push(currentState);
  }
  const container = document.querySelector(containerSelector);
  if (!container) {
    return;
  }
  container.addEventListener('click', handlePanelEvents);
  function handlePanelEvents(event) {
    const target = event.target;
    const clickedHelpIcon = target.closest('.help-icon');
    const allTooltips = container.querySelectorAll('.tooltip-text');
    allTooltips.forEach(tt => {
      if (!tt.parentElement.contains(clickedHelpIcon)) {
        tt.classList.remove('visible');
      }
    });
    if (clickedHelpIcon) {
      const tooltipText = clickedHelpIcon.closest('.tooltip').querySelector('.tooltip-text');
      tooltipText.classList.toggle('visible');
      return;
    }
    if (target.id === 'start-build') {
      resetState();
      state.startingBoardOrientation = board.state.orientation;
      state.color = board.state.orientation;
      state.initialBoardState = {
        fen: game.fen(),
        pgn: game.pgn(),
        orientation: board.state.orientation,
        history: [...game.history()]
      };
      state.fen = game.fen();
      state.savedPgn = game.pgn();
      const openingNameElement = document.getElementById('opening-name-text');
      state.openingName = openingNameElement ? openingNameElement.textContent.trim() : '';
      trackStateChange("start-build clicked");
      goToStep(1);
      return;
    }
    if (target.id === 'back-pos') goToStep(0);
    if (target.id === 'back-color') goToStep(1);
    if (target.id === 'back-mode') goToStep(2);
    if (target.id === 'back-temp') goToStep(3);
    if (target.id === 'confirm-pos') {
      state.fen = game.fen();
      state.savedPgn = game.pgn();
      state.color = board.state.orientation;
      state.initialBoardState = {
        fen: game.fen(),
        pgn: game.pgn(),
        orientation: board.state.orientation,
        history: [...game.history()]
      };
      disableExplorerTab();
      trackStateChange("position confirmed");
      goToStep(2);
    }
    if (target.id === 'confirm-color') {
      const activeColorBtn = document.querySelector('[data-color].active');
      if (activeColorBtn) {
        const selectedColor = activeColorBtn.dataset.color;
        if (state.color !== selectedColor) {
          state.color = selectedColor;
        }
        if (board.state.orientation !== state.color) {
          board.set({ orientation: state.color });
        }
        if (state.initialBoardState) {
          state.initialBoardState.orientation = state.color;
        }
        trackStateChange("color confirmed");
        goToStep(3);
      }
    }
    if (target.id === 'confirm-mode') {
      const activeModeBtn = document.querySelector('[data-mode].active');
      if (activeModeBtn) {
        state.mode = activeModeBtn.dataset.mode;
        trackStateChange("mode confirmed");
        goToStep(4);
      }
    }
    if (target.id === 'confirm-temp') {
      trackStateChange("temperature confirmed");
      const activeTempBtn = document.querySelector('[data-temp-mode].active');
      if (activeTempBtn) {
        state.temperature = activeTempBtn.dataset.tempMode;
        handleGeneration();
      }
    }
    if (target.dataset.color) {
      const previousColor = state.color;
      const newColor = target.dataset.color;
      document.querySelectorAll('[data-color]').forEach(btn => {
        btn.classList.remove('active');
      });
      target.classList.add('active');
      state.color = newColor;
      if (board.state.orientation !== newColor) {
        board.set({ orientation: newColor });
        if (state.initialBoardState) {
          state.initialBoardState.orientation = newColor;
        }
      }
      trackStateChange(`color button ${newColor} clicked`);
    }
    if (target.dataset.mode) {
      document.querySelectorAll('[data-mode]').forEach(btn => {
        btn.classList.remove('active');
      });
      target.classList.add('active');
      state.mode = target.dataset.mode;
      trackStateChange(`mode button ${target.dataset.mode} clicked`);
    }
    if (target.dataset.tempMode) {
      document.querySelectorAll('[data-temp-mode]').forEach(btn => {
        btn.classList.remove('active');
      });
      target.classList.add('active');
      state.temperature = target.dataset.tempMode;
      trackStateChange(`temperature button ${target.dataset.tempMode} clicked`);
    }
    if (target.id === 'generate') {
      handleGeneration();
    }
  }
  function handleGeneration() {
    const config = {
      fen: state.fen,
      openingName: state.openingName,
      color: state.color,
      mode: state.mode,
      temperature: state.temperature,
    };
    trackStateChange("generate clicked");
    disableExplorerTab();
    const buildPanel = document.getElementById('build-panel');
    if (buildPanel) {
      buildPanel.innerHTML = `
          <div class="step-panel-container current">
            <div class="step-panel">
              <div class="generating-indicator">
                Generating
                <span class="dots">
                  <span class="dot"></span>
                  <span class="dot"></span>
                  <span class="dot"></span>
                </span>
              </div>
            </div>
          </div>
        `;
      buildPanel.style.height = 'auto';
      const repertoireResult = document.getElementById('repertoire-result');
      if (repertoireResult) {
        repertoireResult.innerHTML = '';
      }
    }
    generateRepertoire(config).then(generationInitiated => {
      if (generationInitiated) {
        resetState();
      } else {
        goToStep(0, { keepError: true });
      }
    });
  }
  function resetState() {
    state.color = board.state.orientation || 'white';
    state.mode = 'light';
    state.temperature = 'balanced';
    state.fen = '';
    state.openingName = '';
    state.savedPgn = null;
    state.initialBoardState = null;
    state.startingBoardOrientation = null;
    enableExplorerTab();
    trackStateChange("state reset");
  }
  function goToStep(step, { keepError = false } = {}) {
    state.previousStep = state.currentStep;
    state.currentStep = step;
    if (step === 0) {
      enableExplorerTab();
    }
    const boardWrapper = document.getElementById('board').parentElement;
    if (step === 1) {
      if (state.initialBoardState) {
        game.load_pgn(state.initialBoardState.pgn);
        board.set({
          fen: state.initialBoardState.fen,
          orientation: state.initialBoardState.orientation,
          turnColor: toColor(game),
          movable: {
            color: 'both',
            free: false,
            dests: toDests(game)
          },
          lastMove: getLastMove(game)
        });
        updateMainUI(game);
      }
      boardWrapper.classList.remove('board-locked');
      if (toggleBoardInteraction) toggleBoardInteraction(true);
    } else if (step > 1) {
      boardWrapper.classList.add('board-locked');
      board.set({
        movable: {
          color: 'none',
          dests: []
        }
      });
      if (toggleBoardInteraction) toggleBoardInteraction(false);
    } else {
      boardWrapper.classList.remove('board-locked');
      if (toggleBoardInteraction) toggleBoardInteraction(true);
    }
    if (step === 2) {
      state.color = board.state.orientation;
      if (board.state.orientation !== state.color) {
        board.set({ orientation: state.color });
      }
    }
    const repertoireResult = document.getElementById('repertoire-result');
    if (repertoireResult && !keepError) {
      repertoireResult.innerHTML = '';
    }
    trackStateChange(`navigated to step ${step}`);
    updateUI();
  }
  function resetFlow() {
    goToStep(0);
  }
  function updateUI() {
    const buildPanel = document.getElementById('build-panel');
    if (!buildPanel) {
      return;
    }
    if (state.currentStep > 0) {
      buildPanel.classList.add('panel-active');
    } else {
      if (state.initialBoardState) {
        game.load_pgn(state.initialBoardState.pgn);
        board.set({
          fen: state.initialBoardState.fen,
          orientation: state.initialBoardState.orientation,
          turnColor: toColor(game),
          movable: {
            color: toColor(game),
            free: false,
            dests: toDests(game)
          },
          lastMove: getLastMove(game)
        });
        updateMainUI(game);
      }
      buildPanel.classList.remove('panel-active');
    }
    const isForward = state.previousStep !== null && state.currentStep > state.previousStep;
    const isBackward = state.previousStep !== null && state.currentStep < state.previousStep;
    const existingPanels = buildPanel.querySelectorAll('.step-panel-container, .single-button-container');
    if (existingPanels.length > 0) {
      existingPanels.forEach(panel => {
        if (isForward) {
          panel.classList.remove('current');
          panel.classList.add('prev');
        } else {
          panel.classList.remove('current');
          panel.classList.add('next');
        }
        setTimeout(() => {
          if (panel && panel.parentNode) {
            panel.parentNode.removeChild(panel);
          }
        }, 400);
      });
    }
    const delay = existingPanels.length > 0 ? 50 : 0;
    setTimeout(() => {
      let newPanelContent = '';
      let newPanelContainer;
      if (state.currentStep === 0) {
        newPanelContainer = document.createElement('div');
        newPanelContainer.className = 'single-button-container';
        newPanelContainer.innerHTML = `
          <button id="start-build" class="w-full btn btn-lg btn-primary">
            Start Building Repertoire
          </button>
        `;
      } else {
        switch (state.currentStep) {
          case 1:
            newPanelContent = `
              <h3>Choose Starting Position</h3>
              <p>Make moves on the board to set your starting position</p>
              <div class="actions">
                <button id="confirm-pos" class="btn btn-primary">Confirm</button>
                <button id="back-pos" class="btn btn-outline-secondary">Go Back</button>
              </div>
            `;
            break;
          case 2:
            newPanelContent = `
              <h3>Generate Repertoire For:</h3>
              <div class="controls">
                <button class="btn btn-outline-primary ${state.color === 'white' ? 'active' : ''}" data-color="white">White</button>
                <button class="btn btn-outline-primary ${state.color === 'black' ? 'active' : ''}" data-color="black">Black</button>
              </div>
              <div class="actions">
                <button id="confirm-color" class="btn btn-primary">Confirm</button>
                <button id="back-color" class="btn btn-outline-secondary">Go Back</button>
              </div>
            `;
            break;
          case 3:
            newPanelContent = `
              <div class="label-container">
                <h3>Analysis Mode:</h3>
                <div class="tooltip">
                  <span class="help-icon">?</span>
                  <div class="tooltip-text">
                    <strong style="color: #4a77e5;">Light:</strong> covers ~70% of lines found in the database.<br>
                    <strong style="color: #ffb74d;">Deep:</strong> covers ~85% of lines, providing a larger repertoire.<br>
                    <strong style="color: #8a2be2;">Ultradeep:</strong> covers ~95% of lines.
                  </div>
                </div>
              </div>
              <div class="controls">
                <button data-mode="light" class="btn btn-outline-primary ${state.mode === 'light' ? 'active' : ''}">Light</button>
                <button data-mode="deep" class="btn btn-outline-primary ${state.mode === 'deep' ? 'active' : ''}">Deep</button>
                <button data-mode="ultradeep" class="btn btn-outline-primary ${state.mode === 'ultradeep' ? 'active' : ''}">Ultradeep</button>
              </div>
              <div class="actions">
                <button id="confirm-mode" class="btn btn-primary">Confirm</button>
                <button id="back-mode" class="btn btn-outline-secondary">Go Back</button>
              </div>
            `;
            break;
          case 4:
            newPanelContent = `
              <div class="label-container">
                <h3>Temperature:</h3>
                <div class="tooltip">
                  <span class="help-icon">?</span>
                  <div class="tooltip-text">
                    <strong style="color: #a8d8ff;">Solid:</strong> Focuses on popular moves with good win rates<br>
                    <strong style="color: #ffa726;">Balanced:</strong> Mix of popularity and win rate<br>
                    <strong style="color: #ff5252;">Risky:</strong> Focuses on win rate, may include rare moves
                  </div>
                </div>
              </div>
              <div class="controls">
                <button data-temp-mode="solid" class="btn btn-outline-primary ${state.temperature === 'solid' ? 'active' : ''}">Solid</button>
                <button data-temp-mode="balanced" class="btn btn-outline-primary ${state.temperature === 'balanced' ? 'active' : ''}">Balanced</button>
                <button data-temp-mode="risky" class="btn btn-outline-primary ${state.temperature === 'risky' ? 'active' : ''}">Risky</button>
              </div>
              <div class="actions">
                <button id="confirm-temp" class="btn btn-primary">Generate</button>
                <button id="back-temp" class="btn btn-outline-secondary">Go Back</button>
              </div>
            `;
            break;
        }
        newPanelContainer = document.createElement('div');
        newPanelContainer.className = 'step-panel-container';
        const stepPanel = document.createElement('div');
        stepPanel.className = 'step-panel';
        stepPanel.innerHTML = newPanelContent;
        newPanelContainer.appendChild(stepPanel);
      }
      const tempDiv = document.createElement('div');
      tempDiv.style.position = 'absolute';
      tempDiv.style.visibility = 'hidden';
      tempDiv.style.width = buildPanel.offsetWidth + 'px';
      if (state.currentStep > 0) {
        tempDiv.classList.add('panel-active');
      }
      const heightProbe = newPanelContainer.cloneNode(true);
      heightProbe.style.position = 'static';
      tempDiv.appendChild(heightProbe);
      document.body.appendChild(tempDiv);
      const newHeight = tempDiv.offsetHeight;
      document.body.removeChild(tempDiv);
      buildPanel.style.height = `${newHeight}px`;
      if (isBackward) {
        newPanelContainer.classList.add('prev');
      } else {
        newPanelContainer.classList.add('next');
      }
      buildPanel.appendChild(newPanelContainer);
      void newPanelContainer.offsetWidth;
      setTimeout(() => {
        newPanelContainer.classList.add('current');
        if (isBackward) {
          newPanelContainer.classList.remove('prev');
        } else {
          newPanelContainer.classList.remove('next');
        }
      }, 10);
      if (state.currentStep === 4) {
        setTimeout(() => {
          document.querySelectorAll('[data-temp-mode]').forEach(btn => {
            btn.addEventListener('click', () => {
              document.querySelectorAll('[data-temp-mode]').forEach(b => b.classList.remove('active'));
              btn.classList.add('active');
              state.temperature = btn.dataset.tempMode;
              trackStateChange(`temperature button ${btn.dataset.tempMode} clicked (via listener)`);
            });
          });
        }, 20);
      }
      if (state.currentStep === 2) {
        document.querySelectorAll('[data-color]').forEach(btn => {
          const isActive = btn.dataset.color === state.color;
          btn.classList.toggle('active', isActive);
          if (isActive && btn.dataset.color === 'black') {
            setTimeout(() => {
              if (!btn.classList.contains('active')) {
                btn.classList.add('active');
              }
              const whiteBtn = document.querySelector('[data-color="white"]');
              if (whiteBtn && whiteBtn.classList.contains('active')) {
                whiteBtn.classList.remove('active');
              }
            }, 50);
          }
        });
      }
      if (state.currentStep === 3) {
        document.querySelectorAll('[data-mode]').forEach(btn => {
          btn.classList.toggle('active', btn.dataset.mode === state.mode);
        });
      }
    }, delay);
  }
  trackStateChange("initialization");
  goToStep(0);
  return {
    reset: () => {
      goToStep(0);
    },
    getCurrentState: () => {
      const currentState = { ...state };
      delete currentState._stateHistory;
      return currentState;
    },
    resetFlow: () => {
      resetFlow();
    },
    updateBoardReference: (newBoard) => {
      board = newBoard;
    },
    getStateHistory: () => [...state._stateHistory],
    showStartButton: () => {
      resetState();
      const buildPanel = document.getElementById('build-panel');
      if (buildPanel) {
        buildPanel.innerHTML = '';
        buildPanel.classList.remove('panel-active');
        const buttonContainer = document.createElement('div');
        buttonContainer.className = 'single-button-container current';
        buttonContainer.innerHTML = `
          <button id="start-build" class="w-full btn btn-lg btn-primary">
            Start Building Repertoire
          </button>
        `;
        buildPanel.appendChild(buttonContainer);
        const boardWrapper = document.getElementById('board').parentElement;
        boardWrapper.classList.remove('board-locked');
        if (toggleBoardInteraction) toggleBoardInteraction(true);
        state.currentStep = 0;
        enableExplorerTab();
        trackStateChange("start button shown");
      }
    }
  };
} 