// Logic for importing PGN's from the sidebar

let selectedColor = null;
let pgnContent = '';
let errorMessageDiv = null;
let csrfToken = null;
export function initImportPgn() {
    const importPgnModal = document.getElementById('import-pgn-modal');
    const importPgnBtn = document.getElementById('import-pgn-btn');
    const closeImportModal = document.getElementById('close-import-modal');
    const cancelImportBtn = document.getElementById('cancel-import-btn');
    const submitImportBtn = document.getElementById('submit-import-btn');
    if (!importPgnModal || !importPgnBtn) return;
    const repertoireNameInput = document.getElementById('repertoire-name-input');
    const colorButtons = document.querySelectorAll('.color-btn');
    const uploadFileBtn = document.getElementById('upload-file-btn');
    const pgnFileInput = document.getElementById('pgn-file-input');
    const fileNameDisplay = document.getElementById('file-name-display');
    const pgnTextInput = document.getElementById('pgn-text-input');
    importPgnBtn.addEventListener('click', async () => {
        importPgnModal.style.display = 'flex';
        resetImportForm(repertoireNameInput, pgnTextInput, pgnFileInput, fileNameDisplay, colorButtons);
        if (window.innerWidth <= 768) {
            const sidebar = document.getElementById('repertoire-sidebar');
            const mainContent = document.getElementById('main-content');
            const boardWrapper = document.querySelector('.board-wrapper');
            if (sidebar && sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
                if (mainContent) mainContent.classList.remove('sidebar-active');
                if (boardWrapper) boardWrapper.classList.remove('board-disabled');
            }
        }
        if (checkIfUserLoggedIn() && !csrfToken) {
            await fetchCsrfToken();
        }
    });
    if (closeImportModal) closeImportModal.addEventListener('click', closeModal);
    if (cancelImportBtn) cancelImportBtn.addEventListener('click', closeModal);
    window.addEventListener('click', (e) => {
        if (e.target === importPgnModal) closeModal();
    });
    function closeModal() {
        importPgnModal.style.display = 'none';
        if (window.innerWidth <= 768) {
            const boardWrapper = document.querySelector('.board-wrapper');
            if (boardWrapper) {
                boardWrapper.classList.remove('board-disabled');
                boardWrapper.classList.remove('board-locked');
            }
            const mainContent = document.getElementById('main-content');
            if (mainContent) mainContent.classList.remove('sidebar-active');
        }
    }
    colorButtons.forEach(btn => {
        btn.addEventListener('click', () => {
            colorButtons.forEach(b => b.classList.remove('selected'));
            btn.classList.add('selected');
            selectedColor = btn.dataset.color;
        });
    });
    if (uploadFileBtn && pgnFileInput) {
        uploadFileBtn.addEventListener('click', () => pgnFileInput.click());
        pgnFileInput.addEventListener('change', (e) => {
            const file = e.target.files[0];
            if (file) {
                const maxSize = 2 * 1024 * 1024;
                if (file.size > maxSize) {
                    showError('File size exceeds 2MB limit. Please choose a smaller file.');
                    pgnFileInput.value = '';
                    fileNameDisplay.textContent = 'No file chosen';
                    return;
                }
                fileNameDisplay.textContent = file.name;
                clearError();
                const reader = new FileReader();
                reader.onload = (event) => {
                    pgnContent = event.target.result;
                    if (pgnTextInput) pgnTextInput.value = '';
                };
                reader.onerror = () => {
                    showError('Error reading file. Please try again.');
                    pgnFileInput.value = '';
                    fileNameDisplay.textContent = 'No file chosen';
                };
                reader.readAsText(file);
            } else {
                fileNameDisplay.textContent = 'No file chosen';
            }
        });
    }
    if (pgnTextInput) {
        pgnTextInput.addEventListener('input', (e) => {
            pgnContent = e.target.value;
            if (pgnContent.trim() && pgnFileInput) {
                pgnFileInput.value = '';
                if (fileNameDisplay) fileNameDisplay.textContent = 'No file chosen';
            }
        });
    }
    if (submitImportBtn) {
        submitImportBtn.addEventListener('click', async () => {
            const name = repertoireNameInput.value.trim();
            const pgn = pgnContent.trim();
            clearError();
            if (!name) return showError('Please enter a repertoire name.');
            if (!selectedColor) return showError('Please select a color (White or Black).');
            if (!pgn) return showError('Please provide PGN content.');
            if (!validatePgnBasic(pgn)) return showError('Invalid PGN format.');
            submitImportBtn.disabled = true;
            submitImportBtn.textContent = 'Importing...';
            try {
                await saveImportedRepertoire(name, pgn, selectedColor);
                closeModal();
                showSuccess('Repertoire imported successfully!');
            } catch (error) {
                console.error('Import error:', error);
                showError(error.message || 'Failed to import repertoire.');
            } finally {
                submitImportBtn.disabled = false;
                submitImportBtn.textContent = 'Import';
            }
        });
    }
}
function resetImportForm(nameInput, textInput, fileInput, fileDisplay, colorBtns) {
    if (nameInput) nameInput.value = '';
    if (textInput) textInput.value = '';
    if (fileInput) fileInput.value = '';
    if (fileDisplay) fileDisplay.textContent = 'No file chosen';
    pgnContent = '';
    selectedColor = null;
    colorBtns.forEach(b => b.classList.remove('selected'));
    clearError();
}
function validatePgnBasic(pgn) {
    if (pgn.length < 10) return false;
    const hasMoves = /\d+\.\s*[a-hNBRQKO]/.test(pgn);
    const hasHeaders = /\[.*\]/.test(pgn);
    return hasMoves || hasHeaders;
}
async function saveImportedRepertoire(name, pgn, color) {
    const isLoggedIn = checkIfUserLoggedIn();
    const uniqueName = await getUniqueName(name, isLoggedIn);
    if (isLoggedIn) {
        await saveToServer(uniqueName, pgn, color);
    } else {
        saveToLocalStorage(uniqueName, pgn, color);
    }
    if (typeof window.renderRepertoireList === 'function') {
        await window.renderRepertoireList();
    }
}
async function getUniqueName(baseName, isLoggedIn) {
    const existingNames = await getExistingRepertoireNames(isLoggedIn);
    let uniqueName = baseName;
    let counter = 1;
    while (existingNames.includes(uniqueName)) {
        uniqueName = `${baseName} (${counter})`;
        counter++;
    }
    return uniqueName;
}
async function getExistingRepertoireNames(isLoggedIn) {
    if (isLoggedIn) {
        try {
            const response = await fetch('/api/repertoires', { credentials: 'include' });
            if (response.ok) {
                const repertoires = await response.json();
                return repertoires.map(r => r.name);
            }
        } catch (error) {
            console.error('Error fetching repertoires:', error);
        }
        return [];
    } else {
        const repertoires = JSON.parse(localStorage.getItem('saved_repertoires') || '[]');
        return repertoires.map(r => r.name);
    }
}
function checkIfUserLoggedIn() {
    return window.currentUser && window.currentUser.logged_in;
}
async function fetchCsrfToken() {
    try {
        const response = await fetch('/api/csrf-token', { credentials: 'include' });
        if (response.ok) {
            const data = await response.json();
            csrfToken = data.csrf_token;
        }
    } catch (error) {
        console.error('Failed to fetch CSRF token:', error);
    }
}
async function saveToServer(name, pgn, color) {
    if (!csrfToken) await fetchCsrfToken();
    const headers = { 'Content-Type': 'application/json' };
    if (csrfToken) headers['X-CSRF-Token'] = csrfToken;
    const response = await fetch('/api/repertoires', {
        method: 'POST',
        headers: headers,
        credentials: 'include',
        body: JSON.stringify({ name, pgn, color })
    });
    if (!response.ok) {
        let errorMessage = 'Failed to save repertoire';
        try {
            const error = await response.json();
            errorMessage = error.error || errorMessage;
        } catch (e) {
            console.error('Raw response error:', e);
        }
        throw new Error(errorMessage);
    }
    return await response.json();
}
function saveToLocalStorage(name, pgn, color) {
    let repertoires = JSON.parse(localStorage.getItem('saved_repertoires') || '[]');
    const newRepertoire = {
        id: Date.now().toString(),
        name,
        pgn,
        color,
        created: Date.now()
    };
    repertoires.unshift(newRepertoire);
    localStorage.setItem('saved_repertoires', JSON.stringify(repertoires));
}
function showError(message) {
    clearError();
    errorMessageDiv = document.createElement('div');
    errorMessageDiv.className = 'import-error-message';
    errorMessageDiv.textContent = message;
    const formActions = document.querySelector('.form-actions');
    if (formActions) formActions.parentNode.insertBefore(errorMessageDiv, formActions);
}
function clearError() {
    if (errorMessageDiv && errorMessageDiv.parentNode) {
        errorMessageDiv.parentNode.removeChild(errorMessageDiv);
        errorMessageDiv = null;
    }
}
function showSuccess(message) {
    const successDiv = document.createElement('div');
    successDiv.className = 'import-success-message';
    successDiv.textContent = message;
    successDiv.style.cssText = 'position: fixed; top: 20px; right: 20px; background: #4caf50; color: white; padding: 1rem 1.5rem; border-radius: 6px; z-index: 10000; box-shadow: 0 4px 12px rgba(0,0,0,0.3);';
    document.body.appendChild(successDiv);
    setTimeout(() => {
        successDiv.style.opacity = '0';
        successDiv.style.transition = 'opacity 0.3s';
        setTimeout(() => {
            if (successDiv.parentNode) successDiv.parentNode.removeChild(successDiv);
        }, 300);
    }, 2000);
}
if (document.currentScript && !document.currentScript.type === 'module') {
    document.addEventListener('DOMContentLoaded', initImportPgn);
}
