// Main engine for the generator page, managing board, ECO database and repertoire lists

import { Chessground } from "./lib/chessground.min.js";
import { initBuildFlow } from "./buildRepertoireFlow.js";
import { initImportPgn } from "./importPgn.js";
import { initAuth } from "./auth.js";
let csrfToken = null;
document.addEventListener("DOMContentLoaded", () => {
  "use strict";
  const game = new Chess();
  let ground;
  let boardOrientation = "white";
  let redoStack = [];
  let ecoDatabase = null;
  let popularPositionsCache = null;
  let optimizedEcoDb = null;
  let lastOpeningName = "-";
  let lastOpeningEco = "ECO";
  let repertoireFlow = null;
  let toDests = null;
  let isHandlingMoveClick = false;
  window.boardInteractionEnabled = true;
  const statsCache = new Map();
  const MAX_CACHE_SIZE = 100;
  let newRepertoiresCount = 0;
  const MAX_NEW_COUNT = 99;
  let mostRecentRepertoireId = null;
  const boardElement = document.getElementById("board");
  const statsContainer = document.getElementById("stats-container");
  const repertoireResult = document.getElementById("repertoire-result");
  const moveListContainer = document.getElementById("move-list-container");
  const tabs = document.querySelectorAll(".tab-link");
  const flipBoardBtn = document.getElementById("flip-board");
  const backMoveBtn = document.getElementById("back-move");
  const forwardMoveBtn = document.getElementById("forward-move");
  const openingNameText = document.getElementById("opening-name-text");
  const openingEcoCode = document.getElementById("opening-eco-code");
  const sidebar = document.getElementById("repertoire-sidebar");
  const sidebarToggleBtn = document.getElementById("sidebar-toggle");
  const repertoireListWhite = document.getElementById("repertoire-list-white");
  const repertoireListBlack = document.getElementById("repertoire-list-black");
  const mainContent = document.getElementById("main-content");
  const buildPanel = document.getElementById("build-panel");
  const undoBtn = document.getElementById("undo-btn");
  const resetBtn = document.getElementById("reset-btn");
  async function loadDatabases() {
    try {
    } catch (error) {
    }
  }
  let isLoadingEco = false;
  async function loadEcoDatabase() {
    if (optimizedEcoDb !== null || isLoadingEco) {
      return;
    }
    isLoadingEco = true;
    try {
      const ecoResponse = await fetch("/static/data/eco.json");
      if (!ecoResponse.ok) throw new Error(`HTTP error! status: ${ecoResponse.status}`);
      const ecoData = await ecoResponse.json();
      optimizedEcoDb = new Map();
      for (const key in ecoData) {
        if (Object.hasOwnProperty.call(ecoData, key)) {
          const fenParts = key.split(" ");
          const prefix = `${fenParts[0]} ${fenParts[1]} ${fenParts[2]}`;
          if (!optimizedEcoDb.has(prefix)) {
            optimizedEcoDb.set(prefix, ecoData[key]);
          }
        }
      }
    } catch (error) {
      console.error("Failed to load ECO database:", error);
    } finally {
      isLoadingEco = false;
    }
  }
  let isLoadingPopularPositions = false;
  async function loadPopularPositions() {
    if (popularPositionsCache !== null || isLoadingPopularPositions) {
      return;
    }
    isLoadingPopularPositions = true;
    try {
      const popularResponse = await fetch("/static/data/popular_positions.json");
      if (!popularResponse.ok) throw new Error(`HTTP error! status: ${popularResponse.status}`);
      popularPositionsCache = await popularResponse.json();
    } catch (error) {
      console.error("Failed to load popular positions:", error);
    } finally {
      isLoadingPopularPositions = false;
    }
  }
  function chessToDests(chess) {
    const dests = new Map();
    chess.SQUARES.forEach((s) => {
      const ms = chess.moves({ square: s, verbose: true });
      if (ms.length)
        dests.set(
          s,
          ms.map((m) => m.to),
        );
    });
    return dests;
  }
  function toColor(chess) {
    return chess.turn() === "w" ? "white" : "black";
  }
  function getLastMove(chess) {
    const history = chess.history({ verbose: true });
    if (history.length === 0) return undefined;
    const lastMove = history[history.length - 1];
    return [lastMove.from, lastMove.to];
  }
  function handleMoveClick(moveSan) {
    if (isHandlingMoveClick) {
      return;
    }
    isHandlingMoveClick = true;
    const move = game.move(moveSan, { sloppy: true });
    if (move === null) {
      isHandlingMoveClick = false;
      return;
    }
    ground.set({
      fen: game.fen(),
      turnColor: toColor(game),
      movable: {
        color: toColor(game),
        dests: toDests(game),
      },
      lastMove: [move.from, move.to],
    });
    redoStack = [];
    updateUI(game);
    setTimeout(() => {
      isHandlingMoveClick = false;
    }, 400);
  }
  function playOtherSide(chess) {
    return (orig, dest) => {
      chess.move({ from: orig, to: dest, promotion: "q" });
      redoStack = [];
      ground.set({
        fen: chess.fen(),
        turnColor: toColor(chess),
        movable: {
          color: toColor(chess),
          dests: toDests(chess),
        },
      });
      setTimeout(() => {
        updateMoveList(chess);
        updateOpeningName(chess);
        repertoireResult.innerHTML = "";
        setTimeout(() => {
        }, 100);
      }, 0);
    };
  }
  function updateUI(chess, options = {}) {
    const config = {
      fen: chess.fen(),
      turnColor: toColor(chess),
      movable: {
        color: toColor(chess),
        dests: toDests(chess),
      },
      lastMove: getLastMove(chess),
    };
    if (options.fullReset) {
      config.drawable = {
        shapes: [],
      };
      config.lastMove = [];
    }
    if (options.orientation) {
      config.orientation = options.orientation;
      boardOrientation = options.orientation;
    }
    ground.set(config);
    boardOrientation = ground.state.orientation;
    const boardWrapper = document.getElementById("board").parentElement;
    if (boardWrapper) {
      boardWrapper._chessground = ground;
    }
    updateMoveList(chess);
    updateOpeningName(chess);
    const explorerTab = document.getElementById("explorer-tab");
    if (explorerTab && explorerTab.classList.contains("active")) {
      setTimeout(() => {
        fetchStats(chess.fen());
      }, 10);
    }
  }
  function initializeBoard() {
    if (
      game.fen() !== "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1"
    ) {
      game.reset();
    }
    if (ground) {
      ground.destroy();
    }
    ground = Chessground(document.getElementById("board"), {
      movable: {
        free: false,
        color: "white",
        dests: chessToDests(game),
      },
      draggable: {
        enabled: true,
        autoDistance: true,
        showGhost: true,
      },
      orientation: boardOrientation,
      selectable: {
        enabled: true,
      },
      events: {
        move: onMove,
        change: onChange,
      },
      coordinates: true,
    });
    const boardWrapper = document.getElementById("board").parentElement;
    if (boardWrapper) {
      boardWrapper._chessground = ground;
    }
    ground.set({
      movable: {
        dests: chessToDests(game),
      },
    });
    function onMove(orig, dest) {
      const promotion =
        game.get(orig).type === "p" &&
          ((dest[1] === "8" && game.turn() === "w") ||
            (dest[1] === "1" && game.turn() === "b"))
          ? "q"
          : undefined;
      const move = game.move({ from: orig, to: dest, promotion });
      if (move === null) {
        return false;
      }
      if (redoStack.length > 0) {
        redoStack = [];
      }
      ground.set({
        fen: game.fen(),
        turnColor: toColor(game),
        movable: {
          dests: chessToDests(game),
        },
        check: game.in_check(),
      });
      updateUI(game);
      return true;
    }
    function onChange() {
    }
    if (!toDests) {
      toDests = chessToDests;
    }
    updateMoveList(game);
    updateOpeningName(game);
  }
  async function fetchStats(fen) {
    const fenParts = fen.split(" ");
    const cacheKey = `${fenParts[0]} ${fenParts[1]}`;
    if (popularPositionsCache && popularPositionsCache[cacheKey]) {
      const stats = parseLichessData(popularPositionsCache[cacheKey]);
      displayStats(stats);
      return;
    }
    if (statsCache.has(cacheKey)) {
      displayStats(statsCache.get(cacheKey));
      return;
    }
    statsContainer.innerHTML = '<div class="stats-table-container"><p>Loading statistics...</p></div>';
    try {
      const encodedFen = encodeURIComponent(fen);
      const response = await fetch(`/api/fen/${encodedFen}`);
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.description || `HTTP error! status: ${response.status}`);
      }
      const rawStats = await response.json();
      const stats = parseLichessData(rawStats);
      statsCache.set(cacheKey, stats);
      limitCacheSize(statsCache);
      displayStats(stats);
    } catch (error) {
      statsContainer.innerHTML = `<p style="color: #ff8a80;">Error: ${error.message}</p>`;
    }
  }
  function parseLichessData(data) {
    if (!data) {
      return [];
    }
    const moves = [];
    const moveData = data.m || data;
    if (typeof moveData !== 'object' || moveData === null) {
      return [];
    }
    for (const [san, stats] of Object.entries(moveData)) {
      if (stats && typeof stats === 'object' && 'c' in stats) {
        moves.push({
          move: san,
          games: stats.c,
          white: stats.o ? stats.o.W : 0,
          draws: stats.o ? stats.o.D : 0,
          black: stats.o ? stats.o.L : 0,
        });
      } else if (stats && typeof stats === 'object' && 'white' in stats) {
        moves.push({
          move: san,
          games: stats.white + stats.draws + stats.black,
          white: stats.white,
          draws: stats.draws,
          black: stats.black,
        });
      }
    }
    return moves.sort((a, b) => b.games - a.games);
  }
  function debounce(func, delay) {
    let timeout;
    return function () {
      const context = this;
      const args = arguments;
      clearTimeout(timeout);
      timeout = setTimeout(() => func.apply(context, args), delay);
    };
  }
  function displayStats(stats) {
    if (!statsContainer) return;
    statsContainer.innerHTML = "";
    if (!stats || stats.length === 0) {
      statsContainer.innerHTML =
        "<p>No game data available for this position.</p>";
      return;
    }
    const totalGames = stats.reduce((sum, move) => sum + move.games, 0);
    const totalWhiteWins = stats.reduce((sum, move) => sum + move.white, 0);
    const totalBlackWins = stats.reduce((sum, move) => sum + move.black, 0);
    const totalDraws = totalGames - totalWhiteWins - totalBlackWins;
    const whiteWinRate = (totalWhiteWins / totalGames) * 100;
    const drawRate = (totalDraws / totalGames) * 100;
    const blackWinRate = (totalBlackWins / totalGames) * 100;
    const whiteLabel =
      whiteWinRate > 10
        ? `<span class="bar-label">${whiteWinRate.toFixed(1)}%</span>`
        : "";
    const drawLabel =
      drawRate > 10
        ? `<span class="bar-label">${drawRate.toFixed(1)}%</span>`
        : "";
    const blackLabel =
      blackWinRate > 10
        ? `<span class="bar-label">${blackWinRate.toFixed(1)}%</span>`
        : "";
    const summaryHtml = `
            <div class="stats-summary">
                <div class="summary-total">Total Games: ${totalGames.toLocaleString()}</div>
                <div class="summary-bar">
                    <div class="bar-white" style="width: ${whiteWinRate}%" title="White wins: ${whiteWinRate.toFixed(1)}%">${whiteLabel}</div>
                    <div class="bar-draw" style="width: ${drawRate}%" title="Draws: ${drawRate.toFixed(1)}%">${drawLabel}</div>
                    <div class="bar-black" style="width: ${blackWinRate}%" title="Black wins: ${blackWinRate.toFixed(1)}%">${blackLabel}</div>
                </div>
            </div>
        `;
    let tableHtml = `
            <table class="stats-table">
                <thead>
                    <tr>
                        <th>Move</th>
                        <th class="games-header">Games</th>
                        <th class="bar-header">Win Rate</th>
                    </tr>
                </thead>
                <tbody>
        `;
    for (const move of stats) {
      const whiteWinRate = (move.white / move.games) * 100;
      const drawRate = (move.draws / move.games) * 100;
      const blackWinRate = (move.black / move.games) * 100;
      const whiteLabel =
        whiteWinRate > 10
          ? `<span class="bar-label">${whiteWinRate.toFixed(1)}%</span>`
          : "";
      const blackLabel =
        blackWinRate > 10
          ? `<span class="bar-label">${blackWinRate.toFixed(1)}%</span>`
          : "";
      tableHtml += `
                <tr class="move-row">
                    <td class="stats-table-move" data-san="${move.move}">${sanToHtml(move.move)}</td>
                    <td class="games-cell">${move.games.toLocaleString()}</td>
                    <td>
                        <div class="percentage-bar">
                            <div class="bar-white" style="width: ${whiteWinRate}%">${whiteLabel}</div>
                            <div class="bar-draw" style="width: ${drawRate}%"></div>
                            <div class="bar-black" style="width: ${blackWinRate}%">${blackLabel}</div>
                        </div>
                    </td>
                </tr>
            `;
    }
    tableHtml += "</tbody></table>";
    const tempDiv = document.createElement("div");
    tempDiv.innerHTML = summaryHtml + tableHtml;
    statsContainer.appendChild(tempDiv);
    const moveCells = statsContainer.querySelectorAll(".stats-table-move");
    moveCells.forEach((cell) => {
      const row = cell.closest("tr");
      if (row) {
        row.style.cursor = "pointer";
      }
    });
  }
  function sanToHtml(san) {
    return san;
  }
  let previousTopMovesHtml = "";
  function updateMoveList(chess) {
    const moves = chess.history();
    if (moves.length === 0) {
      moveListContainer.innerHTML = "<p>Starting position</p>";
      centerMoveListScrollbar();
      return;
    }
    let moveText = "";
    moves.forEach((move, index) => {
      if (index % 2 === 0) {
        moveText += `${index / 2 + 1}. ${move} `;
      } else {
        moveText += `${move} `;
      }
    });
    moveListContainer.innerHTML = `<p>${moveText}</p>`;
  }
  async function updateOpeningName(chess) {
    if (chess.history().length === 0) {
      lastOpeningName = "-";
      lastOpeningEco = "ECO";
    } else {
      await loadEcoDatabase();
      if (!optimizedEcoDb) return;
      const fen = chess.fen();
      const fenParts = fen.split(" ");
      const board = fenParts[0];
      const turn = fenParts[1];
      const castling = fenParts[2];
      const lookupPrefix = `${board} ${turn} ${castling}`;
      const opening = optimizedEcoDb.get(lookupPrefix);
      if (opening && opening.name) {
        lastOpeningName = opening.name;
        lastOpeningEco = opening.eco || "ECO";
      }
    }
    openingNameText.innerText = lastOpeningName;
    openingEcoCode.innerText = lastOpeningEco;
  }
  function showRepertoireError(message) {
    repertoireResult.innerHTML = `
            <div class="error">
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="margin-right: 8px; vertical-align: middle;">
                    <circle cx="12" cy="12" r="10"></circle>
                    <line x1="12" y1="8" x2="12" y2="12"></line>
                    <line x1="12" y1="16" x2="12.01" y2="16"></line>
                </svg>
                ${message}
            </div>
        `;
  }
  function updateNewRepertoiresCounter() {
    const counter = document.getElementById("new-repertoires-counter");
    const mobileCounter = document.getElementById("mobile-repertoires-counter");
    if (counter) {
      if (newRepertoiresCount > 0) {
        counter.textContent = newRepertoiresCount;
        counter.classList.add("visible");
      } else {
        counter.classList.remove("visible");
      }
    }
    if (mobileCounter) {
      if (newRepertoiresCount > 0) {
        mobileCounter.textContent = newRepertoiresCount;
        mobileCounter.classList.add("visible");
      } else {
        mobileCounter.classList.remove("visible");
      }
    }
  }
  async function getServerRepertoires() {
    try {
      const response = await fetch("/api/repertoires");
      if (!response.ok) {
        if (response.status === 401) return [];
        throw new Error("Failed to fetch repertoires from server");
      }
      return await response.json();
    } catch (error) {
      return [];
    }
  }
  async function saveRepertoireToServer(repertoire) {
    try {
      const response = await fetchWithCSRF("/api/repertoires", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          name: repertoire.name,
          pgn: repertoire.pgn,
          color: repertoire.color,
        }),
      });
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(
          errorData.error || "Failed to save repertoire to server",
        );
      }
      const result = await response.json();
      return result.id;
    } catch (error) {
      return false;
    }
  }
  async function fetchWithCSRF(url, options = {}) {
    if (!options.headers) {
      options.headers = {};
    }
    if (csrfToken) {
      options.headers['X-CSRF-Token'] = csrfToken;
    }
    return fetch(url, options);
  }
  async function migrateRepertoires() {
    const localRepertoires = getSavedRepertoires();
    if (localRepertoires.length === 0) return;
    try {
      const response = await fetchWithCSRF("/api/repertoires/migrate", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ repertoires: localRepertoires }),
      });
      if (response.ok) {
        const result = await response.json();
        localStorage.removeItem("saved_repertoires");
        newRepertoiresCount = 0;
        updateNewRepertoiresCounter();
        await renderRepertoireList();
        return true;
      } else {
        const errorData = await response.json();
        return false;
      }
    } catch (error) {
      return false;
    }
  }
  async function deleteRepertoireFromServer(repertoireId) {
    try {
      const response = await fetchWithCSRF(`/api/repertoires/${repertoireId}`, {
        method: "DELETE",
      });
      if (!response.ok) {
        throw new Error("Failed to delete repertoire from server");
      }
      return true;
    } catch (error) {
      return false;
    }
  }
  function getSavedRepertoires() {
    const saved = localStorage.getItem("saved_repertoires");
    const repertoires = saved ? JSON.parse(saved) : [];
    return repertoires;
  }
  function saveRepertoires(repertoires) {
    localStorage.setItem("saved_repertoires", JSON.stringify(repertoires));
  }
  async function saveRepertoire(name, pgn, color) {
    if (!name || !pgn || !color) {
      return;
    }
    if (window.currentUser && window.currentUser.logged_in) {
      const newRepertoire = { name, pgn, color };
      const serverId = await saveRepertoireToServer(newRepertoire);
      if (!serverId) {
        return;
      }
      mostRecentRepertoireId = serverId;
      newRepertoiresCount++;
      updateNewRepertoiresCounter();
      await renderRepertoireList();
      return serverId;
    } else {
      const repertoires = getSavedRepertoires();
      let finalName = name;
      let counter = 1;
      while (repertoires.some((r) => r.name === finalName)) {
        finalName = `${name} (${counter})`;
        counter++;
      }
      const repertoireId = Date.now().toString();
      const newRepertoire = {
        id: repertoireId,
        name: finalName,
        pgn,
        color,
        created: Date.now(),
      };
      repertoires.unshift(newRepertoire);
      saveRepertoires(repertoires);
      mostRecentRepertoireId = repertoireId;
      newRepertoiresCount++;
      updateNewRepertoiresCounter();
      await renderRepertoireList();
      return finalName;
    }
  }
  async function deleteRepertoire(repertoire) {
    if (window.currentUser && window.currentUser.logged_in) {
      const success = await deleteRepertoireFromServer(repertoire.id);
      if (!success) return;
    } else {
      let repertoires = getSavedRepertoires();
      repertoires = repertoires.filter((rep) => rep.name !== repertoire.name);
      saveRepertoires(repertoires);
    }
    await renderRepertoireList();
  }
  async function renderRepertoireList() {
    let repertoires = [];
    if (window.currentUser && window.currentUser.logged_in) {
      repertoires = await getServerRepertoires();
    } else {
      repertoires = getSavedRepertoires();
      repertoires.sort((a, b) => (b.created || 0) - (a.created || 0));
    }
    repertoireListWhite.innerHTML = "";
    repertoireListBlack.innerHTML = "";
    if (repertoires.length === 0) {
      repertoireListWhite.innerHTML =
        '<li class="empty-list">No saved white repertoires.</li>';
      repertoireListBlack.innerHTML =
        '<li class="empty-list">No saved black repertoires.</li>';
      return;
    }
    let whiteRepertoiresCount = 0;
    let blackRepertoiresCount = 0;
    repertoires.forEach((rep) => {
      const listItem = document.createElement("li");
      if (rep.id && rep.id === mostRecentRepertoireId) {
        listItem.classList.add("recent-repertoire");
      }
      const nameSpan = document.createElement("span");
      nameSpan.className = "repertoire-name";
      nameSpan.textContent = rep.name;
      nameSpan.title = rep.name;
      const actionsDiv = document.createElement("div");
      actionsDiv.className = "repertoire-actions";
      const trainBtn = document.createElement("button");
      trainBtn.className = "train-btn";
      trainBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><polyline points="12 6 12 12 16 14"></polyline></svg>`;
      trainBtn.title = "Train Repertoire";
      trainBtn.addEventListener("click", (e) => {
        e.stopPropagation();
        const repertoireIdentifier =
          window.currentUser && window.currentUser.logged_in
            ? rep.id
            : rep.name;
        window.location.href = `/training?repertoire=${encodeURIComponent(repertoireIdentifier)}`;
      });
      const downloadBtn = document.createElement("button");
      downloadBtn.className = "download-btn";
      downloadBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path><polyline points="7 10 12 15 17 10"></polyline><line x1="12" y1="15" x2="12" y2="3"></line></svg>`;
      downloadBtn.title = "Download Repertoire";
      downloadBtn.addEventListener("click", (e) => {
        e.stopPropagation();
        downloadRepertoire(rep.pgn || rep.pgn_data, rep.name, downloadBtn);
      });
      const deleteBtn = document.createElement("button");
      deleteBtn.className = "delete-btn";
      deleteBtn.innerHTML = "&#10005;";
      deleteBtn.title = "Delete Repertoire";
      deleteBtn.addEventListener("click", async (e) => {
        e.stopPropagation();
        if (confirm(`Are you sure you want to delete "${rep.name}"?`)) {
          await deleteRepertoire(rep);
        }
      });
      actionsDiv.appendChild(trainBtn);
      actionsDiv.appendChild(downloadBtn);
      actionsDiv.appendChild(deleteBtn);
      listItem.appendChild(nameSpan);
      listItem.appendChild(actionsDiv);
      const isMobile = () => window.innerWidth <= 768;
      if (isMobile()) {
        listItem.addEventListener("click", (e) => {
          if (e.target.closest("button")) return;
          const isExpanded = listItem.classList.toggle("expanded");
          if (isExpanded) {
            document
              .querySelectorAll(
                "#repertoire-sidebar .repertoire-list li.expanded",
              )
              .forEach((li) => {
                if (li !== listItem) li.classList.remove("expanded");
              });
            loadRepertoireIntoBoard(rep.pgn || rep.pgn_data);
          }
        });
      } else {
        listItem.addEventListener("mouseenter", () =>
          listItem.classList.add("expanded"),
        );
        listItem.addEventListener("mouseleave", () =>
          listItem.classList.remove("expanded"),
        );
        listItem.addEventListener("click", (e) => {
          if (e.target.closest("button")) return;
          loadRepertoireIntoBoard(rep.pgn || rep.pgn_data);
        });
      }
      if (rep.color === "white") {
        repertoireListWhite.appendChild(listItem);
        whiteRepertoiresCount++;
      } else {
        repertoireListBlack.appendChild(listItem);
        blackRepertoiresCount++;
      }
    });
    if (whiteRepertoiresCount === 0) {
      repertoireListWhite.innerHTML =
        '<li class="empty-list">No saved white repertoires.</li>';
    }
    if (blackRepertoiresCount === 0) {
      repertoireListBlack.innerHTML =
        '<li class="empty-list">No saved black repertoires.</li>';
    }
  }
  window.renderRepertoireList = renderRepertoireList;
  function loadRepertoireIntoBoard(pgn) {
    game.load_pgn(pgn);
    const newFen = game.fen();
    ground.set({
      fen: newFen,
      turn: toColor(game),
      dests: toDests(game),
      lastMove: getLastMove(game),
      movable: {
        color: toColor(game),
        dests: toDests(game),
      },
    });
    updateUI(game);
    const explorerTab = document.getElementById("explorer-tab");
    if (explorerTab && explorerTab.classList.contains("active")) {
      setTimeout(() => {
        fetchStats(game.fen());
      }, 100);
    }
  }
  function toggleSidebar() {
    sidebar.classList.toggle("active");
    mainContent.classList.toggle("sidebar-active");
    const isActive = sidebar.classList.contains("active");
    const boardWrapper = document.querySelector(".board-wrapper");
    if (boardWrapper) {
      if (isActive) {
        boardWrapper.classList.add("board-disabled");
      } else {
        boardWrapper.classList.remove("board-disabled");
      }
    }
    if (window.innerWidth <= 768) {
      if (sidebar) {
        if (!isActive && getComputedStyle(sidebar).display === "flex") {
          setTimeout(() => {
            if (!sidebar.classList.contains("active")) {
              sidebar.style.display = "none";
            }
          }, 300);
        } else if (isActive) {
          sidebar.style.display = "flex";
        }
      }
    }
    if (isActive) {
      newRepertoiresCount = 0;
      updateNewRepertoiresCounter();
    }
    sidebarToggleBtn.setAttribute("aria-expanded", isActive);
    const mobileMenuBtn = document.getElementById("mobile-menu-btn");
    if (mobileMenuBtn) {
      mobileMenuBtn.setAttribute("aria-expanded", isActive);
    }
  }
  function downloadRepertoire(pgn, filename, buttonElement = null) {
    if (buttonElement) {
      buttonElement.classList.add("downloading");
      setTimeout(() => buttonElement.classList.remove("downloading"), 500);
    }
    setTimeout(() => {
      const blob = new Blob([pgn], { type: "application/x-chess-pgn" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = filename + '.pgn';
      a.style.display = "none";
      document.body.appendChild(a);
      setTimeout(() => {
        a.click();
        setTimeout(() => {
          document.body.removeChild(a);
          URL.revokeObjectURL(url);
        }, 100);
      }, 50);
    }, 0);
  }
  function handleUndo() {
    if (game.history().length > 0) {
      const undoneMove = game.undo();
      if (undoneMove) {
        redoStack.push(undoneMove);
        ground.set({
          fen: game.fen(),
          turnColor: toColor(game),
          movable: {
            color: toColor(game),
            dests: toDests(game),
          },
          lastMove: getLastMove(game),
        });
        setTimeout(() => {
          updateMoveList(game);
          updateOpeningName(game);
          repertoireResult.innerHTML = "";
          const explorerTab = document.getElementById("explorer-tab");
          if (explorerTab && explorerTab.classList.contains("active")) {
            fetchStats(game.fen());
          }
        }, 0);
      }
    }
  }
  function handleRedo() {
    const move = redoStack.pop();
    if (move) {
      game.move(move);
      ground.set({
        fen: game.fen(),
        turnColor: toColor(game),
        movable: {
          color: toColor(game),
          dests: toDests(game),
        },
        lastMove: [move.from, move.to],
      });
      setTimeout(() => {
        updateMoveList(game);
        updateOpeningName(game);
        repertoireResult.innerHTML = "";
        const explorerTab = document.getElementById("explorer-tab");
        if (explorerTab && explorerTab.classList.contains("active")) {
          fetchStats(game.fen());
        }
      }, 0);
    }
  }
  function handleReset() {
    game.reset();
    redoStack = [];
    clearAllCaches();
    updateUI(game, { fullReset: true });
    toggleBoardInteraction(true);
  }
  function clearAllCaches() {
    statsCache.clear();
  }
  async function handleTabClick(e) {
    const targetTab = e.target.dataset.tab;
    document
      .querySelectorAll(".tab-content")
      .forEach((c) => c.classList.remove("active"));
    document
      .querySelectorAll(".tab-link")
      .forEach((l) => l.classList.remove("active"));
    document.getElementById(targetTab).classList.add("active");
    e.target.classList.add("active");
    if (targetTab === "explorer-tab") {
      await loadPopularPositions();
      fetchStats(game.fen());
    }
  }
  function handleSliderInput() {
    const displayElement =
      this.id === "depth-slider" ? depthValue : temperatureValue;
    if (displayElement) {
      displayElement.textContent = this.value;
    }
  }
  function clearStatsCache() {
    statsCache.clear();
  }
  function limitCacheSize(cache) {
    if (cache.size > MAX_CACHE_SIZE) {
      const keysToRemove = Array.from(cache.keys()).slice(
        0,
        Math.floor(MAX_CACHE_SIZE * 0.2),
      );
      keysToRemove.forEach((key) => cache.delete(key));
    }
  }
  function handleFlipBoard() {
    boardOrientation = boardOrientation === "white" ? "black" : "white";
    ground.set({ orientation: boardOrientation });
    const boardWrapper = document.getElementById("board").parentElement;
    if (boardWrapper) {
      boardWrapper._chessground = ground;
    }
    if (repertoireFlow) {
      const state = repertoireFlow.getCurrentState();
      if (state.currentStep > 0) {
        state.color = boardOrientation;
        if (state.currentStep === 2) {
          const whiteBtn = document.querySelector('[data-color="white"]');
          const blackBtn = document.querySelector('[data-color="black"]');
          if (whiteBtn && blackBtn) {
            if (boardOrientation === "white") {
              whiteBtn.classList.add("active");
              blackBtn.classList.remove("active");
            } else {
              whiteBtn.classList.remove("active");
              blackBtn.classList.add("active");
            }
          }
        }
      }
    }
  }
  function toggleBoardInteraction(enable) {
    if (ground) {
      if (enable) {
        const currentFen = game.fen();
        ground.set({
          fen: currentFen,
          movable: {
            color: toColor(game),
            free: false,
            dests: toDests(game),
          },
          turnColor: toColor(game),
          lastMove: getLastMove(game),
        });
        const boardWrapper = document.querySelector(".board-wrapper");
        if (boardWrapper) {
          boardWrapper.classList.remove("board-locked");
        }
      } else {
        ground.set({
          movable: {
            color: "none",
            free: false,
            dests: [],
          },
          premovable: {
            enabled: false,
          },
          draggable: {
            enabled: false,
          },
        });
        const boardWrapper = document.querySelector(".board-wrapper");
        if (boardWrapper) {
          boardWrapper.classList.add("board-locked");
        }
      }
    }
    const controlButtons = document.querySelectorAll(
      ".board-controls .board-btn",
    );
    controlButtons.forEach((btn) => {
      if (enable) {
        btn.disabled = false;
        btn.classList.remove("disabled");
      } else {
        btn.disabled = true;
        btn.classList.add("disabled");
      }
    });
    window.boardInteractionEnabled = enable;
  }
  function fullBoardReset() {
    game.reset();
    redoStack = [];
    clearAllCaches();
    if (ground) {
      ground.destroy();
    }
    initializeBoard();
    if (repertoireFlow) {
      repertoireFlow.updateBoardReference(ground);
    }
    updateMoveList(game);
    updateOpeningName(game);
    if (repertoireResult) {
      repertoireResult.innerHTML = "";
    }
    toggleBoardInteraction(true);
  }
  async function pollTaskStatus(taskId, repName, color) {
    const startTime = Date.now();
    const timeout = 60000;
    const intervalId = setInterval(async () => {
      if (Date.now() - startTime > timeout) {
        clearInterval(intervalId);
        showRepertoireError("Repertoire generation timed out. Please try again.");
        toggleBoardInteraction(true);
        repertoireFlow.resetFlow();
        return;
      }
      try {
        const response = await fetch(`/api/repertoire/status/${taskId}`);
        if (!response.ok) {
          return;
        }
        const data = await response.json();
        if (data.state === 'SUCCESS') {
          clearInterval(intervalId);
          toggleBoardInteraction(true);
          const result = data.result;
          if (result && result.pgn) {
            const serverId = await saveRepertoire(repName, result.pgn, color);
            const repertoireResultEl = document.getElementById('repertoire-result');
            const buildPanelEl = document.getElementById('build-panel');
            if (repertoireResultEl && buildPanelEl) {
              repertoireFlow.resetFlow();
              const trainBtn = document.createElement('button');
              trainBtn.className = 'train-new-btn';
              trainBtn.innerHTML = 'Train Repertoire <span class="beta-badge">Beta</span>';
              trainBtn.onclick = () => {
                if (serverId) {
                  window.location.href = `/training?repertoire=${encodeURIComponent(serverId)}`;
                } else {
                }
              };
              repertoireResultEl.innerHTML = '';
              repertoireResultEl.appendChild(trainBtn);
            }
          } else {
            showRepertoireError("Failed to process the generated repertoire. Unexpected format.");
            repertoireFlow.resetFlow();
          }
        } else if (data.state === 'FAILURE') {
          clearInterval(intervalId);
          showRepertoireError(data.result.error || "Repertoire generation failed on the server.");
          toggleBoardInteraction(true);
          repertoireFlow.resetFlow();
        }
      } catch (error) {
      }
    }, 2000);
  }
  async function startAsyncGeneration(config) {
    const { color, mode, temperature, openingName } = config;
    const moves = game.history();
    try {
      const response = await fetchWithCSRF("/api/repertoire/generate", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          moves: moves.join(" "),
          color: color,
          temperature_mode: temperature,
          analysis_width: mode,
        }),
      });
      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.error || `Server error (${response.status})`);
      }
      const data = await response.json();
      if (data.task_id) {
        let nameBase = (openingName && openingName !== "-") ? openingName : (moves.length > 0 ? moves.slice(0, 3).join("_") : "starting_position");
        const sanitizedName = nameBase.replace(/[:']/g, "").replace(/\s/g, "_");
        const repName = `${sanitizedName}_${config.mode}_${config.temperature}_${config.color}`;
        await pollTaskStatus(data.task_id, repName, config.color);
        return true;
      } else {
        throw new Error("Server did not return a task ID.");
      }
    } catch (error) {
      showRepertoireError(error.message || "Failed to start generation.");
      toggleBoardInteraction(true);
      repertoireFlow.resetFlow();
      return false;
    }
  }
  async function generateRepertoire(config) {
    const { fen } = config;
    toggleBoardInteraction(false);
    const moves = game.history();
    if (moves.length > 14) {
      showRepertoireError("Starting position exceeds the limit of 7 full moves.");
      toggleBoardInteraction(true);
      return false;
    }
    try {
      const encodedFen = encodeURIComponent(fen);
      const response = await fetch(`/api/fen/${encodedFen}`);
      if (!response.ok) {
        showRepertoireError("Could not validate position frequency.");
        toggleBoardInteraction(true);
        return false;
      }
      const stats = await response.json();
      const totalGames = Object.values(stats.m || {}).reduce((sum, move) => sum + move.c, 0);
      if (totalGames < 1500) {
        showRepertoireError(`Position is too rare (${totalGames.toLocaleString()} games found).`);
        toggleBoardInteraction(true);
        return false;
      }
    } catch (error) {
      showRepertoireError("An error occurred during position validation.");
      toggleBoardInteraction(true);
      return false;
    }
    return await startAsyncGeneration(config);
  }
  function initializeEventListeners() {
    if (flipBoardBtn) flipBoardBtn.addEventListener("click", handleFlipBoard);
    if (backMoveBtn) backMoveBtn.addEventListener("click", handleUndo);
    if (forwardMoveBtn) forwardMoveBtn.addEventListener("click", handleRedo);
    if (sidebarToggleBtn) {
      sidebarToggleBtn.addEventListener("click", (e) => {
        e.stopPropagation();
        toggleSidebar();
      });
    }
    const mobileMenuBtn = document.getElementById("mobile-menu-btn");
    if (mobileMenuBtn) {
      mobileMenuBtn.addEventListener("click", (e) => {
        e.stopPropagation();
        toggleSidebar();
        newRepertoiresCount = 0;
        updateNewRepertoiresCounter();
      });
    }
    const googleLoginBtn = document.getElementById("google-login-btn");
    if (googleLoginBtn) {
      googleLoginBtn.addEventListener("click", (e) => {
        e.stopPropagation();
      });
    }
    tabs.forEach((tab) => {
      tab.addEventListener("click", handleTabClick);
    });
    const depthSlider = document.getElementById("depth-slider");
    const depthValue = document.getElementById("depth-value");
    if (depthSlider && depthValue) {
      depthSlider.value = 4;
      depthValue.textContent = 4;
      depthSlider.addEventListener("input", handleSliderInput);
    }
    const temperatureSlider = document.getElementById("temperature-slider");
    const temperatureValue = document.getElementById("temperature-value");
    if (temperatureSlider && temperatureValue) {
      temperatureSlider.addEventListener("input", () => {
        temperatureValue.textContent = temperatureSlider.value;
      });
    }
    let navigationThrottled = false;
    const throttleNavigation = (callback) => {
      if (!navigationThrottled) {
        navigationThrottled = true;
        callback();
        setTimeout(() => {
          navigationThrottled = false;
        }, 100);
      }
    };
    document.addEventListener("keydown", (e) => {
      if (window.boardInteractionEnabled === false) {
        return;
      }
      if (e.key === "ArrowLeft") {
        e.preventDefault();
        throttleNavigation(handleUndo);
      } else if (e.key === "ArrowRight") {
        e.preventDefault();
        throttleNavigation(handleRedo);
      }
    });
    boardElement.addEventListener(
      "wheel",
      (e) => {
        if (window.boardInteractionEnabled === false) {
          return;
        }
        e.preventDefault();
        if (e.deltaY < 0) {
          throttleNavigation(handleUndo);
        } else {
          throttleNavigation(handleRedo);
        }
      },
      { passive: false },
    );
    mainContent.addEventListener("click", () => {
      if (sidebar.classList.contains("active")) {
        toggleSidebar();
      }
    });
    if (statsContainer) {
      statsContainer.addEventListener("click", (e) => {
        const row = e.target.closest(".stats-table tr");
        if (!row) {
          return;
        }
        const moveCell = row.querySelector(".stats-table-move[data-san]");
        if (moveCell) {
          const moveSan = moveCell.dataset.san;
          e.preventDefault();
          handleMoveClick(moveSan);
        } else {
          const allTds = row.querySelectorAll("td");
        }
      });
    }
    if (repertoireListWhite) {
      repertoireListWhite.addEventListener("click", (e) => {
        const listItem = e.target.closest("li");
        if (listItem && listItem.classList.contains("empty-list")) {
          return;
        }
        const repName = listItem
          ? listItem.querySelector(".repertoire-name").textContent
          : null;
        if (repName) {
          const repertoires = getSavedRepertoires();
          const rep = repertoires.find((r) => r.name === repName);
          if (rep) {
            game.load_pgn(rep.pgn);
            const newFen = game.fen();
            boardOrientation = rep.color || "white";
            ground.set({
              fen: newFen,
              turn: toColor(game),
              orientation: boardOrientation,
              dests: toDests(game),
              lastMove: getLastMove(game),
              movable: {
                color: toColor(game),
                dests: toDests(game),
              },
            });
            updateUI(game);
          }
        }
      });
    }
    if (repertoireListBlack) {
      repertoireListBlack.addEventListener("click", (e) => {
        const listItem = e.target.closest("li");
        if (listItem && listItem.classList.contains("empty-list")) {
          return;
        }
        const repName = listItem
          ? listItem.querySelector(".repertoire-name").textContent
          : null;
        if (repName) {
          const repertoires = getSavedRepertoires();
          const rep = repertoires.find((r) => r.name === repName);
          if (rep) {
            game.load_pgn(rep.pgn);
            const newFen = game.fen();
            ground.set({
              fen: newFen,
              turn: toColor(game),
              dests: toDests(game),
              lastMove: getLastMove(game),
              movable: {
                color: toColor(game),
                dests: toDests(game),
              },
            });
            updateUI(game);
          }
        }
      });
    }
  }
  function updateAuthUI(userData) {
    const userInfoMobile = document.getElementById('user-info-mobile');
    if (userData && userData.logged_in) {
      let username = userData.username || 'User';
      const originalUsername = username;
      if (username.length > 9) {
        username = username.substring(0, 9) + '...';
      }
      userInfoMobile.innerHTML = `
            <div class="user-info">
                <div class="username" title="${originalUsername}">${username}</div>
                <button class="logout-btn">Logout</button>
            </div>
        `;
      userInfoMobile.querySelector('.logout-btn').addEventListener('click', () => {
        window.location.href = '/logout';
      });
    } else {
      userInfoMobile.innerHTML = `
            <button class="sign-in-btn" id="sign-in-btn-mobile">
                Sign In
            </button>
        `;
      document.getElementById('sign-in-btn-mobile').addEventListener('click', () => {
        document.getElementById('auth-modal').style.display = 'block';
      });
    }
  }
  async function checkAuthAndMigrate() {
    try {
      const response = await fetchWithCSRF("/api/me");
      if (response.ok) {
        const userData = await response.json();
        window.currentUser = userData;
        if (userData.logged_in) {
          updateAuthUI(userData);
          const localRepertoires = getSavedRepertoires();
          if (localRepertoires.length > 0) {
            const migrationSuccess = await migrateRepertoires();
            if (migrationSuccess) {
            } else {
            }
          } else {
            await renderRepertoireList();
          }
        } else {
          window.currentUser = { logged_in: false };
        }
      } else {
        window.currentUser = { logged_in: false };
      }
    } catch (error) {
      window.currentUser = { logged_in: false };
    }
  }
  async function fetchCsrfToken() {
    try {
      const response = await fetch('/api/csrf-token');
      const data = await response.json();
      csrfToken = data.csrf_token;
    } catch (error) {
    }
  }
  async function runInitialSetup() {
    await loadDatabases();
    initializeBoard();
    initializeCookieBanner();
    repertoireFlow = initBuildFlow({
      containerSelector: "#build-panel",
      board: ground,
      game: game,
      generateRepertoire,
      updateMainUI: updateUI,
      toDests: toDests,
      toggleBoardInteraction,
      toColor: toColor,
      getLastMove: getLastMove,
    });
    initializeEventListeners();
    renderRepertoireList();
    setupClickableMoves();
    setupScrollPrevention();
    centerMoveListScrollbar();
    updateMoveList(game);
    updateUI(game);
    updateNewRepertoiresCounter();
    await fetchCsrfToken();
    checkAuthAndMigrate();
  }
  function centerMoveListScrollbar() {
    if (moveListContainer) {
      const paragraphs = moveListContainer.querySelectorAll("p");
      if (
        paragraphs.length === 1 &&
        paragraphs[0].textContent.includes("Play moves on the board")
      ) {
        setTimeout(() => {
          const totalHeight = moveListContainer.scrollHeight;
          const visibleHeight = moveListContainer.clientHeight;
          const scrollPosition = (totalHeight - visibleHeight) / 2;
          moveListContainer.scrollTop = scrollPosition;
        }, 100);
      }
    }
  }
  function setupClickableMoves() {
    document.addEventListener("click", function (event) {
      if (event.target.closest(".stats-table tr")) {
        const row = event.target.closest(".stats-table tr");
        const moveCell = row.querySelector(".stats-table-move");
        if (moveCell && moveCell.dataset.san) {
          event.preventDefault();
          event.stopPropagation();
          handleMoveClick(moveCell.dataset.san);
        }
      }
    });
  }
  function preventScrollUpdateUI(chess) {
    if (statsContainer) {
      const currentHeight = statsContainer.offsetHeight;
      statsContainer.style.minHeight = currentHeight + "px";
    }
    if (moveListContainer) {
      const currentHeight = moveListContainer.offsetHeight;
      moveListContainer.style.minHeight = currentHeight + "px";
    }
    updateMoveList(chess);
    updateOpeningName(chess);
    setTimeout(() => {
      if (statsContainer) statsContainer.style.minHeight = "";
      if (moveListContainer) moveListContainer.style.minHeight = "";
    }, 500);
  }
  function setupScrollPrevention() {
    document.documentElement.classList.add("no-scroll-jump");
    document.body.classList.add("no-scroll-jump");
    if ("scrollRestoration" in history) {
      history.scrollRestoration = "manual";
    }
    const scrollSensitiveElements = [
      statsContainer,
      moveListContainer,
    ];
    scrollSensitiveElements.forEach((element) => {
      if (element) {
        element.addEventListener("DOMNodeInserted", function (e) {
          e.stopPropagation();
        });
      }
    });
    const observer = new MutationObserver(function (mutations) {
      const scrollY = window.scrollY;
      setTimeout(() => {
        window.scrollTo(0, scrollY);
      }, 0);
    });
    const explorerTab = document.getElementById("explorer-tab");
    if (explorerTab) {
      observer.observe(explorerTab, {
        childList: true,
        subtree: true,
        attributes: true,
        characterData: true,
      });
    }
  }
  function initializeCookieBanner() {
    const banner = document.getElementById('cookie-banner');
    const acceptBtn = document.getElementById('cookie-accept-btn');
    const declineBtn = document.getElementById('cookie-decline-btn');
    if (!banner || !acceptBtn || !declineBtn) {
      return;
    }
    if (document.cookie.split(';').some((item) => item.trim().startsWith('cookie_consent='))) {
      banner.style.display = 'none';
    } else {
      banner.style.display = 'flex';
    }
    acceptBtn.addEventListener('click', () => {
      setCookieConsent(true);
      banner.style.display = 'none';
    });
    declineBtn.addEventListener('click', () => {
      setCookieConsent(false);
      banner.style.display = 'none';
    });
  }
  function setCookieConsent(consent) {
    const d = new Date();
    d.setTime(d.getTime() + (365 * 24 * 60 * 60 * 1000));
    let expires = "expires=" + d.toUTCString();
    document.cookie = `cookie_consent=${consent};` + expires + ";path=/";
  }
  initializeCookieBanner();
  initAuth();
  initImportPgn();
  runInitialSetup();
});
