// Handles user actions like move clicks, hints and board navigation

import { state } from './state.js';
import * as BOARD from './board.js';
import * as UI from './ui.js';
import { onUserMove, startTrainingSession, selectNextBranch, makeOpponentMove, resetToBranchStart } from './training.js';
import * as DOM from './dom.js';
import { getChessInstance, releaseChessInstances, toColor, toDests } from './utils.js';
import { saveProgress } from './progress.js';
import { createBranches } from './logic.js';
import { updatePgnViewer, updateProgressCounters } from './ui.js';
export function onPgnMoveClick(fen) {
    if (fen) {
        const wasInTrainingMode = !state.isExploreMode;
        if (wasInTrainingMode) {
            if (state.currentBranchIndex !== -1 && state.branches[state.currentBranchIndex]) {
                state.savedTrainingState = {
                    branchIndex: state.currentBranchIndex,
                    progressIndex: state.branches[state.currentBranchIndex].progressIndex,
                    fen: state.game.fen()
                };
            } else {
                state.savedTrainingState = null;
            }
        }
        const shortFen = fen.split(' ')[0];
        const fullFen = state.repertoire.shortFenMap[shortFen];
        if (!fullFen || !state.repertoire.fenMap[fullFen]) {
            state.game.load(fen);
            BOARD.updateBoard();
            return;
        }
        const targetNode = state.repertoire.fenMap[fullFen];
        const moveHistory = [];
        let currentNode = targetNode;
        while (currentNode && currentNode.parentFen) {
            moveHistory.unshift(currentNode.san);
            const parentFullFen = state.repertoire.shortFenMap[currentNode.parentFen.split(' ')[0]];
            if (parentFullFen && state.repertoire.fenMap[parentFullFen]) {
                currentNode = state.repertoire.fenMap[parentFullFen];
            } else {
                currentNode = null;
            }
        }
        if (currentNode && currentNode.san) {
            moveHistory.unshift(currentNode.san);
        }
        state.game.load(state.repertoire.startFen);
        moveHistory.forEach(san => {
            state.game.move(san, { sloppy: true });
        });
        if (wasInTrainingMode) {
            enterExploreMode();
        } else {
            BOARD.updateBoard();
        }
    }
}
export function navigateHalfMove(direction) {
    if (!state.isExploreMode) return;
    if (direction < 0 && state.game.history().length > 0) {
        state.game.undo();
        BOARD.updateBoard();
    } else if (direction > 0) {
        const currentFen = state.game.fen();
        const shortFen = currentFen.split(' ')[0];
        const fullFen = state.repertoire.shortFenMap[shortFen];
        if (fullFen && state.repertoire.fenMap[fullFen]) {
            const node = state.repertoire.fenMap[fullFen];
            if (node && node.children && node.children.length === 1) {
                const moveSan = node.children[0].san;
                state.game.move(moveSan, { sloppy: true });
                BOARD.updateBoard();
            }
        }
    }
}
export function handleKeyPress(e) {
    if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') {
        return;
    }
    if (e.key === 'ArrowRight') {
        e.preventDefault();
        navigateHalfMove(1);
    } else if (e.key === 'ArrowLeft') {
        e.preventDefault();
        navigateHalfMove(-1);
    }
}
export function toggleFullTree() {
    const toggleTreeBtn = document.getElementById('toggle-tree-btn');
    const pgn = document.getElementById('pgn-viewer');
    const isExpanded = toggleTreeBtn.textContent === '-';
    if (isExpanded) {
        toggleTreeBtn.textContent = '+';
        toggleTreeBtn.title = 'Expand Full Tree';
        pgn.querySelectorAll('.ot-toggler').forEach(toggler => {
            if (toggler.textContent === '[-]') {
                toggler.click();
            }
        });
    } else {
        toggleTreeBtn.textContent = '-';
        toggleTreeBtn.title = 'Collapse Full Tree';
        pgn.querySelectorAll('.ot-toggler').forEach(toggler => {
            if (toggler.textContent === '[+]') {
                toggler.click();
            }
        });
    }
}
export function setSubRepertoireLayout(active) {
    if (!DOM.trainingControls || !DOM.goBackToMainBtn) return;
    if (active) {
        DOM.goBackToMainBtn.style.display = 'block';
        DOM.trainingControls.style.display = 'flex';
        DOM.trainingControls.style.gap = '10px';
        DOM.exploreRepertoireBtn.style.width = 'auto';
        DOM.exploreRepertoireBtn.style.flexGrow = '1';
        DOM.goBackToMainBtn.style.flexGrow = '1';
    } else {
        DOM.goBackToMainBtn.style.display = 'none';
        DOM.trainingControls.style.display = 'block';
        DOM.exploreRepertoireBtn.style.width = '100%';
        DOM.exploreRepertoireBtn.style.flexGrow = '0';
    }
}
export function startTrainingFromFen(fen) {
    let foundBranchInfo = null;
    for (let i = 0; i < state.branches.length; i++) {
        const branch = state.branches[i];
        const moveIndex = branch.moves.findIndex(m => m.fen.startsWith(fen.split(' ')[0]));
        if (moveIndex !== -1) {
            foundBranchInfo = { branchIndex: i, moveIndex: moveIndex };
            break;
        }
    }
    if (foundBranchInfo) {
        state.currentBranchIndex = foundBranchInfo.branchIndex;
        const branch = state.branches[state.currentBranchIndex];
        branch.progressIndex = foundBranchInfo.moveIndex + 1;
        state.game.load(fen);
        BOARD.updateBoard();
        UI.forceClearFeedback();
        if (branch.progressIndex >= branch.moves.length) {
            branch.practiced = true;
            saveProgress();
            UI.updateOpeningTreeProgress();
            UI.showFeedback(true);
            setTimeout(selectNextBranch, 1000);
        } else {
            if (toColor(state.game) !== state.playerColor) {
                makeOpponentMove();
            }
        }
    } else {
        state.game.load(fen);
        BOARD.updateBoard();
        state.currentBranchIndex = -1;
    }
}
export function enterExploreMode() {
    if (state.isExploreMode) return;
    if (state.currentBranchIndex !== -1 && state.branches[state.currentBranchIndex]) {
        state.savedTrainingState = {
            branchIndex: state.currentBranchIndex,
            progressIndex: state.branches[state.currentBranchIndex].progressIndex,
            fen: state.game.fen()
        };
    } else {
        state.savedTrainingState = null;
    }
    if (state.originalRepertoire) {
        DOM.goBackToMainBtn.style.display = 'none';
    }
    UI.forceClearFeedback();
    state.isExploreMode = true;
    DOM.exploreRepertoireBtn.style.display = 'none';
    DOM.trainActionsContainer.style.display = 'flex';
    if (DOM.backMoveBtn) DOM.backMoveBtn.style.display = 'inline-block';
    if (DOM.forwardMoveBtn) DOM.forwardMoveBtn.style.display = 'inline-block';
    state.ground.set({
        movable: {
            free: false,
            color: 'both',
            dests: toDests(state.game),
            events: {
                after: (orig, dest) => {
                    if (state.isExploreMode) {
                        try {
                            state.game.move({ from: orig, to: dest, promotion: 'q' });
                            BOARD.updateBoard();
                        } catch (e) {
                        }
                    }
                }
            }
        }
    });
    BOARD.updateBoard();
}
export function exitExploreMode() {
    state.isExploreMode = false;
    DOM.exploreRepertoireBtn.style.display = 'block';
    DOM.trainActionsContainer.style.display = 'none';
    if (DOM.backMoveBtn) DOM.backMoveBtn.style.display = 'none';
    if (DOM.forwardMoveBtn) DOM.forwardMoveBtn.style.display = 'none';
    if (state.originalRepertoire) {
        setSubRepertoireLayout(true);
    }
    if (state.savedTrainingState) {
        state.currentBranchIndex = state.savedTrainingState.branchIndex;
        const branch = state.branches[state.currentBranchIndex];
        branch.progressIndex = state.savedTrainingState.progressIndex;
        state.game.load(state.savedTrainingState.fen);
        state.ground.set({
            movable: {
                free: false,
                color: toColor(state.game),
                dests: toDests(state.game),
                events: { after: onUserMove }
            }
        });
        BOARD.updateBoard();
        if (toColor(state.game) !== state.playerColor && branch.progressIndex < branch.moves.length) {
            setTimeout(makeOpponentMove, 500);
        }
        state.savedTrainingState = null;
    } else {
        state.ground.set({
            movable: {
                free: false,
                color: toColor(state.game),
                dests: toDests(state.game),
                events: { after: onUserMove }
            }
        });
        startTrainingFromFen(state.game.fen());
    }
}
