// Manages the Chessground board for training

import { Chessground } from '../lib/chessground.min.js';
import { showRepertoireArrows, clearRepertoireArrows } from './arrows.js';
import { state } from './state.js';
import { toColor, toDests, getLastMove } from './utils.js';
import { updateOpeningTreeProgress, updateMoveHighlight } from './ui.js';
import { boardElement } from './dom.js';
export function clearSquareMark() {
    const feedbackSvg = document.getElementById('feedback-svg');
    if (feedbackSvg) {
        feedbackSvg.innerHTML = '';
    }
}
export function showSquareMark(square, isCorrect) {
    if (state.isExploreMode) return;
    clearSquareMark();
    const feedbackSvg = document.getElementById('feedback-svg');
    const boardWrap = document.querySelector('.cg-wrap');
    if (!feedbackSvg || !boardWrap) return;
    const bounds = boardWrap.getBoundingClientRect();
    const squareSize = bounds.width / 8;
    const col = square.charCodeAt(0) - 'a'.charCodeAt(0);
    const row = 8 - parseInt(square.charAt(1));
    const flipped = state.ground.state.orientation === 'black';
    const x = flipped ? 7 - col : col;
    const y = flipped ? 7 - row : row;
    const symbol = isCorrect ? '✓' : '✗';
    const color = isCorrect ? 'rgba(21, 120, 27, 0.9)' : 'rgba(170, 32, 32, 0.9)';
    const textColor = 'white';
    const group = document.createElementNS('http://www.w3.org/2000/svg', 'g');
    const radius = squareSize * 0.2;
    const cx = x * squareSize + (squareSize * 0.8);
    const cy = y * squareSize + (squareSize * 0.2);
    const circle = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
    circle.setAttribute('cx', cx);
    circle.setAttribute('cy', cy);
    circle.setAttribute('r', radius);
    circle.setAttribute('fill', color);
    circle.style.stroke = 'rgba(255, 255, 255, 0.3)';
    circle.style.strokeWidth = '2';
    const text = document.createElementNS('http://www.w3.org/2000/svg', 'text');
    text.setAttribute('x', cx);
    text.setAttribute('y', cy);
    text.setAttribute('dy', '0.35em');
    text.setAttribute('font-size', squareSize * 0.3);
    text.setAttribute('font-weight', 'bold');
    text.setAttribute('text-anchor', 'middle');
    text.setAttribute('fill', textColor);
    text.textContent = symbol;
    group.appendChild(circle);
    group.appendChild(text);
    feedbackSvg.appendChild(group);
    if (!isCorrect) {
        group.classList.add('mark-fade-out');
        setTimeout(clearSquareMark, 300);
    }
}
export function updateRepertoireHints() {
    if (!state.repertoire || !state.repertoire.fenMap) return;
    const currentFen = state.game.fen();
    const node = state.repertoire.fenMap[currentFen];
    clearRepertoireArrows(state.ground);
    if (state.isExploreMode) {
        if (node && node.children) {
            showRepertoireArrows(state.ground, node, Infinity);
        }
    } else if (node && node.children && toColor(state.game) === state.playerColor) {
        const untriedMoves = node.children.filter(childNode => {
            return !childNode.value || childNode.value === 0;
        });
        if (untriedMoves.length === 0) {
            return;
        }
        const tempChildren = untriedMoves.map(childNode => {
            return {
                ...childNode,
                displayValue: 0
            };
        });
        showRepertoireArrows(state.ground, { ...node, children: tempChildren }, 2);
    }
}
export function updateBoard() {
    state.ground.set({
        fen: state.game.fen(),
        turnColor: toColor(state.game),
        lastMove: getLastMove(state.game),
        movable: {
            color: state.isExploreMode ? 'both' : toColor(state.game),
            dests: toDests(state.game)
        }
    });
    updateMoveHighlight();
    updateRepertoireHints();
    updateOpeningTreeProgress();
}
export function initializeBoard(callbacks = {}) {
    const config = {
        fen: state.game.fen(),
        turnColor: toColor(state.game),
        viewOnly: false,
        disableContextMenu: true,
        movable: {
            color: 'white',
            free: false,
            dests: toDests(state.game),
            events: {
                after: callbacks.onUserMove
            }
        },
        events: {
            move: callbacks.onBoardMove,
        },
        drawable: {
            enabled: true,
            shapes: [],
            brushes: {
                green: { color: '#15781B', opacity: 1, lineWidth: 10 },
                red: { color: '#882020', opacity: 1, lineWidth: 10 },
                blue: { color: '#003088', opacity: 1, lineWidth: 10 },
                yellow: { color: '#e6b400', opacity: 1, lineWidth: 10 }
            }
        }
    };
    state.ground = Chessground(boardElement, config);
}
export function flipBoard() {
    state.boardOrientation = (state.boardOrientation === 'white') ? 'black' : 'white';
    state.ground.set({ orientation: state.boardOrientation });
}
