// Core training logic for move validation, opponent responses and tree travelling

import { getChessInstance, releaseChessInstances } from './utils.js';
import { state } from './state.js';
export function createBranches(tree, initialFen) {
    const createdBranches = [];
    function buildFromNode(startNode, trunkFen) {
        let currentNode = startNode;
        let pathChess = getChessInstance(trunkFen);
        const moves = [];
        while (currentNode) {
            const moveResult = pathChess.move(currentNode.san, { sloppy: true });
            if (!moveResult) break;
            moves.push({
                san: currentNode.san,
                from: moveResult.from,
                to: moveResult.to,
                fen: pathChess.fen()
            });
            const isFork = currentNode.children.length > 1;
            const isEndOfLine = currentNode.children.length === 0;
            if (isFork || isEndOfLine) {
                createdBranches.push({
                    id: `${trunkFen}|${moves.map(m => m.san).join(',')}`,
                    moves: [...moves],
                    trunkFEN: trunkFen,
                    practiced: false,
                    progressIndex: 0,
                });
            }
            if (isFork) {
                for (const childNode of currentNode.children) {
                    buildFromNode(childNode, pathChess.fen());
                }
                currentNode = null;
            } else {
                currentNode = isEndOfLine ? null : currentNode.children[0];
            }
        }
    }
    tree.forEach(rootNode => buildFromNode(rootNode, initialFen));
    releaseChessInstances();
    return createdBranches;
}
export function addFenToTree(branch, chess) {
    for (const node of branch) {
        const branchChess = getChessInstance(chess.fen());
        const moveResult = branchChess.move(node.san, { sloppy: true });
        if (moveResult) {
            node.fen = branchChess.fen();
            node.value = 0;
            node.updated = null;
            if (node.children.length > 0) {
                addFenToTree(node.children, branchChess);
            }
        }
    }
}
export function buildFenMap(nodes, startFen) {
    const fenMap = {};
    const uniquePositions = new Set();
    const chess1 = getChessInstance(startFen);
    const chess2 = getChessInstance(startFen);
    function identifyUniquePositions(branch, currentChess) {
        const branchAsArray = Array.isArray(branch) ? branch : (branch ? [branch] : []);
        for (const move of branchAsArray) {
            const moveResult = currentChess.move(move.san, { sloppy: true });
            if (moveResult) {
                const fen = currentChess.fen();
                uniquePositions.add(fen);
                if (move.children && move.children.length > 0) {
                    identifyUniquePositions(move.children, currentChess);
                }
                currentChess.undo();
            }
        }
    }
    identifyUniquePositions(nodes, chess1);
    function traverse(branch, currentChess) {
        const branchAsArray = Array.isArray(branch) ? branch : (branch ? [branch] : []);
        for (const move of branchAsArray) {
            const moveResult = currentChess.move(move.san, { sloppy: true });
            if (moveResult) {
                const fen = currentChess.fen();
                const shortFen = fen.split(' ')[0];
                if (uniquePositions.has(fen) && !fenMap[shortFen]) {
                    fenMap[shortFen] = move;
                }
                if (move.children && move.children.length > 0) {
                    traverse(move.children, currentChess);
                }
                currentChess.undo();
            }
        }
    }
    traverse(nodes, chess2);
    releaseChessInstances();
    return fenMap;
}
export function buildComprehensiveFenMap(nodes, startFen) {
    const chess = getChessInstance(startFen);
    state.repertoire.fenMap[startFen] = { children: nodes, fen: startFen };
    state.repertoire.shortFenMap[startFen.split(' ')[0]] = startFen;
    state.fenLookupCache.clear();
    const positionTracker = new Map();
    const nodeQueue = [];
    for (const node of nodes) {
        nodeQueue.push({ node, parentFen: startFen, depth: 0 });
    }
    while (nodeQueue.length > 0) {
        const { node, parentFen, depth } = nodeQueue.shift();
        const currentChess = getChessInstance(parentFen);
        const move = currentChess.move(node.san, { sloppy: true });
        if (move) {
            const nextFen = currentChess.fen();
            const shortFen = nextFen.split(' ')[0];
            if (positionTracker.has(shortFen)) {
                node.isDuplicate = true;
                node.duplicateOf = positionTracker.get(shortFen);
            } else {
                node.isDuplicate = false;
                positionTracker.set(shortFen, node);
                node.from = move.from;
                node.to = move.to;
                node.parentFen = parentFen;
                state.repertoire.fenMap[nextFen] = node;
                if (!state.repertoire.shortFenMap[shortFen]) {
                    state.repertoire.shortFenMap[shortFen] = nextFen;
                }
                state.fenLookupCache.set(shortFen, nextFen);
            }
            for (const child of node.children) {
                nodeQueue.push({ node: child, parentFen: nextFen, depth: depth + 1 });
            }
        }
    }
    const cleanupTree = (branch) => {
        for (let i = branch.length - 1; i >= 0; i--) {
            const node = branch[i];
            if (node.isDuplicate) {
                branch.splice(i, 1);
            } else if (node.children.length > 0) {
                cleanupTree(node.children);
            }
        }
    };
    cleanupTree(state.repertoire.tree);
    releaseChessInstances();
}
