// Parses PGN into a tree structure and renders the interactive move list

function parsePgnToTree(pgnString) {
  const tokens = tokenizePgn(pgnString);
  const root = { children: [] };
  let parentStack = [root];
  const variationStack = [];
  let currentMoveNumber = 1;
  let isWhiteTurn = true;
  for (let i = 0; i < tokens.length; i++) {
    const token = tokens[i];
    if (token === '(') {
      if (parentStack.length > 1) {
        variationStack.push(parentStack);
        parentStack = parentStack.slice(0, -1);
      }
    } else if (token === ')') {
      if (variationStack.length > 0) {
        parentStack = variationStack.pop();
      }
    } else if (token.match(/^\d+\.+$/)) {
      const numStr = token.match(/^(\d+)/)[1];
      currentMoveNumber = parseInt(numStr, 10);
      isWhiteTurn = !token.includes('...');
    } else if (token !== '') {
      const moveNode = {
        moveNumber: currentMoveNumber,
        san: token,
        isWhiteTurn: isWhiteTurn,
        children: []
      };
      const parent = parentStack[parentStack.length - 1];
      parent.children.push(moveNode);
      parentStack.push(moveNode);
      if (isWhiteTurn) {
        isWhiteTurn = false;
      } else {
        isWhiteTurn = true;
        currentMoveNumber++;
      }
    }
  }
  return root.children;
}
function tokenizePgn(pgnString) {
  let cleanPgn = pgnString.replace(/\[.*?\]/g, '')
    .replace(/\{.*?\}/g, '')
    .replace(/1-0|0-1|1\/2-1\/2|\*/g, '');
  cleanPgn = cleanPgn.replace(/\(/g, ' ( ').replace(/\)/g, ' ) ');
  cleanPgn = cleanPgn.replace(/\s+/g, ' ').trim();
  return cleanPgn.split(' ').filter(token => token.length > 0);
}
function renderOpeningTree(container, openingTree, srsData = null, getFragmentIdFn = null, playerColor = 'white') {
  container.innerHTML = '';
  for (const moveNode of openingTree) {
    renderLine(container, moveNode, srsData, getFragmentIdFn, playerColor);
  }
}
function renderLine(container, startNode, srsData = null, getFragmentIdFn = null, playerColor = 'white') {
  const lineEl = document.createElement('div');
  lineEl.className = 'ot-node';
  const movesContainer = document.createElement('span');
  movesContainer.className = 'ot-line-moves';
  let currentNode = startNode;
  let lastNodeInLine = startNode;
  let currentFen = null;
  let movesList = [];
  while (currentNode) {
    const moveEl = document.createElement('span');
    moveEl.className = 'ot-move-text';
    const moveColor = currentNode.isWhiteTurn ? 'white' : 'black';
    if (moveColor !== playerColor) {
      moveEl.classList.add('opponent-move');
    }
    const moveText = currentNode.isWhiteTurn
      ? `${currentNode.moveNumber}. ${currentNode.san}`
      : `${currentNode.moveNumber}... ${currentNode.san}`;
    moveEl.textContent = moveText;
    if (currentNode.fen) {
      currentFen = currentNode.fen;
      moveEl.dataset.fen = currentNode.fen;
      movesList.push(currentNode.san);
      if (srsData && getFragmentIdFn && currentFen) {
        const tempFragment = {
          id: currentFen + '|' + movesList.join(','),
          startFen: currentFen,
          moves: [...movesList]
        };
        const fragmentId = getFragmentIdFn(tempFragment);
        const fragmentData = srsData[fragmentId];
        if (fragmentData) {
          let progressClass = '';
          if (fragmentData.repetitions > 0) {
            const threshold = Math.max(1, Math.floor(5 - fragmentData.ease));
            if (fragmentData.repetitions >= threshold) {
              progressClass = 'ot-move-learned';
            } else {
              progressClass = 'ot-move-learning';
            }
          } else {
            progressClass = 'ot-move-not-started';
          }
          if (progressClass) {
            moveEl.classList.add(progressClass);
          }
        }
      }
    }
    movesContainer.appendChild(moveEl);
    movesContainer.appendChild(document.createTextNode(' '));
    lastNodeInLine = currentNode;
    if (currentNode.children.length === 1) {
      currentNode = currentNode.children[0];
    } else {
      break;
    }
  }
  const hasVariations = lastNodeInLine.children.length > 1;
  const togglerEl = document.createElement('span');
  togglerEl.className = 'ot-toggler';
  lineEl.appendChild(togglerEl);
  lineEl.appendChild(movesContainer);
  if (hasVariations) {
    togglerEl.textContent = '[+]';
    const childrenEl = document.createElement('div');
    childrenEl.className = 'ot-children ot-collapsed';
    togglerEl.addEventListener('click', (e) => {
      e.stopPropagation();
      const isCollapsed = childrenEl.classList.contains('ot-collapsed');
      togglerEl.textContent = isCollapsed ? '[-]' : '[+]';
      childrenEl.classList.toggle('ot-collapsed');
    });
    for (const childNode of lastNodeInLine.children) {
      renderLine(childrenEl, childNode, srsData, getFragmentIdFn, playerColor);
    }
    lineEl.appendChild(childrenEl);
  }
  container.appendChild(lineEl);
}
document.addEventListener('DOMContentLoaded', function () {
  const style = document.createElement('style');
  const css = `
    .ot-node {
      margin-left: 10px;
    }
    .ot-toggler {
      cursor: pointer;
      margin-right: 5px;
      user-select: none;
    }
    .ot-move-text {
      cursor: pointer;
      padding: 2px 4px;
      border-radius: 3px;
    }
    .ot-move-text:hover {
      text-decoration: underline;
    }
    .ot-children {
      margin-left: 20px;
      border-left: 1px solid #ccc;
      padding-left: 10px;
    }
    .ot-variation {
       margin-top: 5px;
       margin-bottom: 5px;
    }
    .ot-collapsed {
      display: none;
    }
    .ot-move-learned {
      background-color: rgba(34, 197, 94, 0.2);
      border-left: 3px solid #22c55e !important;
    }
    .ot-move-learning {
      background-color: rgba(234, 179, 8, 0.2);
      border-left: 3px solid #eab308 !important;
    }
    .ot-move-not-started {
      background-color: rgba(100, 116, 139, 0.1);
      border-left: 3px solid #64748b !important;
    }
    .ot-move-text.active {
      background-color: rgba(59, 130, 246, 0.5) !important;
      color: white;
      border-left: 3px solid #1d4ed8 !important;
    }
  `;
  style.textContent = css;
  document.head.appendChild(style);
});
export { parsePgnToTree, renderOpeningTree };
