// Handles loading and processing repertoire data from PGN or JSON formats

import { parsePgnToTree } from './openingTree.js';
import { state } from './state.js';
import * as DOM from './dom.js';
import { startTrainingSession } from './training.js';
import { buildComprehensiveFenMap, createBranches } from './logic.js';
import { loadProgress } from './progress.js';
import * as UI from './ui.js';
import { flipBoard, updateBoard } from './board.js';
import { getChessInstance, releaseChessInstances } from './utils.js';
import * as ACTIONS from './actions.js';
import { onUserMove } from './training.js';
import { LEARNED_THRESHOLD } from './constants.js';
export function getSavedRepertoires() {
    const saved = localStorage.getItem('saved_repertoires');
    return saved ? JSON.parse(saved) : [];
}
export async function renderRepertoireList() {
    let repertoires = [];
    if (window.currentUser && window.currentUser.logged_in) {
        try {
            const response = await fetch('/api/repertoires');
            if (response.ok) {
                repertoires = await response.json();
            }
        } catch (error) {
        }
    } else {
        repertoires = getSavedRepertoires();
        repertoires.sort((a, b) => (b.created || b.id) - (a.created || a.id));
    }
    DOM.repertoireListWhite.innerHTML = '';
    DOM.repertoireListBlack.innerHTML = '';
    if (repertoires.length === 0) {
        DOM.repertoireListWhite.innerHTML = '<li class="empty-list">No saved white repertoires.</li>';
        DOM.repertoireListBlack.innerHTML = '<li class="empty-list">No saved black repertoires.</li>';
        return;
    }
    let whiteCount = 0, blackCount = 0;
    repertoires.forEach(rep => {
        const listItem = document.createElement('li');
        const nameSpan = document.createElement('span');
        nameSpan.className = 'repertoire-name';
        nameSpan.textContent = rep.name;
        nameSpan.title = rep.name;
        const actionsDiv = document.createElement('div');
        actionsDiv.className = 'repertoire-actions';
        const trainBtn = document.createElement('button');
        trainBtn.className = 'train-btn';
        trainBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><polyline points="12 6 12 12 16 14"></polyline></svg>`;
        trainBtn.title = 'Train Repertoire';
        trainBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            const identifier = (window.currentUser && window.currentUser.logged_in) ? rep.id : rep.name;
            window.location.href = `?repertoire=${encodeURIComponent(identifier)}`;
        });
        const downloadBtn = document.createElement('button');
        downloadBtn.className = 'download-btn';
        downloadBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path><polyline points="7 10 12 15 17 10"></polyline><line x1="12" y1="15" x2="12" y2="3"></line></svg>`;
        downloadBtn.title = 'Download Repertoire';
        downloadBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            downloadRepertoire(rep.pgn || rep.pgn_data, rep.name, downloadBtn);
        });
        const deleteBtn = document.createElement('button');
        deleteBtn.className = 'delete-btn';
        deleteBtn.innerHTML = '&#10005;';
        deleteBtn.title = 'Delete Repertoire';
        deleteBtn.addEventListener('click', async (e) => {
            e.stopPropagation();
            if (confirm(`Are you sure you want to delete "${rep.name}"?`)) {
                await deleteRepertoire(rep);
            }
        });
        actionsDiv.appendChild(trainBtn);
        actionsDiv.appendChild(downloadBtn);
        actionsDiv.appendChild(deleteBtn);
        listItem.appendChild(nameSpan);
        listItem.appendChild(actionsDiv);
        const isMobile = () => window.innerWidth <= 768;
        if (isMobile()) {
            listItem.addEventListener('click', (e) => {
                if (e.target.closest('button')) return;
                const isExpanded = listItem.classList.toggle('expanded');
                if (isExpanded) {
                    document.querySelectorAll('#repertoire-sidebar .repertoire-list li.expanded').forEach(li => {
                        if (li !== listItem) li.classList.remove('expanded');
                    });
                }
            });
        } else {
            listItem.addEventListener('mouseenter', () => listItem.classList.add('expanded'));
            listItem.addEventListener('mouseleave', () => listItem.classList.remove('expanded'));
        }
        if (rep.color === 'black') {
            DOM.repertoireListBlack.appendChild(listItem);
            blackCount++;
        } else {
            DOM.repertoireListWhite.appendChild(listItem);
            whiteCount++;
        }
    });
    if (whiteCount === 0) {
        DOM.repertoireListWhite.innerHTML = '<li class="empty-list">No saved white repertoires.</li>';
    }
    if (blackCount === 0) {
        DOM.repertoireListBlack.innerHTML = '<li class="empty-list">No saved black repertoires.</li>';
    }
}
export function loadRepertoireFromData(pgn, name, color) {
    try {
        DOM.openingNameElement.textContent = "Loading...";
        if (DOM.pgnViewerElement) {
            DOM.pgnViewerElement.innerHTML = '';
            DOM.pgnViewerElement.classList.add('loading');
            DOM.pgnViewerElement.style.textAlign = 'center';
            DOM.pgnViewerElement.textContent = 'Loading...';
        }
        const openingTree = parsePgnToTree(pgn);
        if (!openingTree || openingTree.length === 0) {
            throw new Error("Could not parse any valid moves from the PGN.");
        }
        const tempChess = new Chess();
        try { tempChess.load_pgn(pgn); } catch (e) { }
        const startFenHeader = tempChess.header().FEN;
        const startFen = startFenHeader ? startFenHeader : 'rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1';
        state.playerColor = color || 'white';
        state.repertoire = {
            name: name,
            pgn: pgn,
            tree: openingTree,
            startFen: startFen,
            playerColor: state.playerColor,
            fenMap: {},
            shortFenMap: {},
            learnedNodes: new Set()
        };
        DOM.openingNameElement.textContent = name;
        buildComprehensiveFenMap(state.repertoire.tree, state.repertoire.startFen);
        state.branches = createBranches(state.repertoire.tree, state.repertoire.startFen);
        loadProgress();
        UI.updateProgressCounters();
        if (state.playerColor !== state.boardOrientation) {
            flipBoard();
        }
        UI.updatePgnViewer(state.repertoire.tree, state.repertoire.startFen);
        const progressLegend = document.querySelector('.progress-legend');
        if (progressLegend) progressLegend.style.display = 'block';
        if (DOM.feedbackContainerElement) DOM.feedbackContainerElement.style.display = 'flex';
        if (DOM.exploreRepertoireBtn) DOM.exploreRepertoireBtn.style.display = 'block';
        const noRepertoireMessage = document.getElementById('no-repertoire-message');
        if (noRepertoireMessage) noRepertoireMessage.style.display = 'none';
        const trainingHeader = document.querySelector('.training-header');
        if (trainingHeader) trainingHeader.style.display = 'block';
        startTrainingSession();
        return true;
    } catch (e) {
        UI.showError(`Failed to load repertoire: ${e.message}`);
        state.repertoire = null;
        return false;
    }
}
export function saveRepertoire(name, pgn, color) {
    if (!name || !pgn || !color) {
        UI.showError("Cannot save. Name, PGN, and color are required.");
        return false;
    }
    try {
        const repertoires = getSavedRepertoires();
        const existingIndex = repertoires.findIndex(r => r.name === name);
        if (existingIndex !== -1) {
            repertoires[existingIndex] = { name, pgn, color };
        } else {
            repertoires.unshift({ name, pgn, color });
        }
        localStorage.setItem('saved_repertoires', JSON.stringify(repertoires));
        return true;
    } catch (e) {
        UI.showError(`Error saving repertoire: ${e.message}`);
        return false;
    }
}
export function importPgnForTraining(pgnText, name) {
    if (!pgnText || !name) {
        UI.showError("Please provide both a name and PGN data.");
        return false;
    }
    const tempChess = getChessInstance();
    try {
        if (!tempChess.load_pgn(pgnText)) throw new Error("Invalid PGN format.");
    } catch (e) {
        UI.showError(`Invalid PGN: ${e.message}`);
        releaseChessInstances();
        return false;
    }
    releaseChessInstances();
    const color = prompt("Is this repertoire for 'white' or 'black'?", "white");
    if (color !== 'white' && color !== 'black') {
        UI.showError("Invalid color. Please enter 'white' or 'black'.");
        return false;
    }
    if (saveRepertoire(name, pgnText, color)) {
        renderRepertoireList();
        window.location.href = `?repertoire=${encodeURIComponent(name)}`;
        return true;
    }
    return false;
}
export async function deleteRepertoire(repertoire) {
    const isGuest = !(window.currentUser && window.currentUser.logged_in);
    if (isGuest) {
        let repertoires = getSavedRepertoires();
        repertoires = repertoires.filter(r => r.name !== repertoire.name);
        localStorage.setItem('saved_repertoires', JSON.stringify(repertoires));
    } else {
        try {
            const response = await fetch(`/api/repertoires/${repertoire.id}`, {
                method: 'DELETE',
                headers: {
                }
            });
            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || 'Failed to delete repertoire from server');
            }
        } catch (error) {
            UI.showError("Could not delete repertoire. Please try again.");
            return;
        }
    }
    const urlParams = new URLSearchParams(window.location.search);
    const currentIdentifier = urlParams.get('repertoire');
    const deletedIdentifier = isGuest ? repertoire.name : repertoire.id.toString();
    if (currentIdentifier === deletedIdentifier) {
        window.location.href = window.location.pathname;
    } else {
        renderRepertoireList();
    }
}
export function downloadRepertoire(pgn, filename, buttonElement = null) {
    if (buttonElement) {
        buttonElement.classList.add('downloading');
        setTimeout(() => buttonElement.classList.remove('downloading'), 500);
    }
    setTimeout(() => {
        const blob = new Blob([pgn], { type: 'application/x-chess-pgn' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = filename.endsWith('.pgn') ? filename : `${filename}.pgn`;
        a.style.display = 'none';
        document.body.appendChild(a);
        setTimeout(() => {
            a.click();
            setTimeout(() => {
                document.body.removeChild(a);
                URL.revokeObjectURL(url);
            }, 100);
        }, 50);
    }, 0);
}
export function loadSamplePGN() {
    const samplePgn = `[Event "vs Computer"]
[Site "Chess.com"]
[PlyCount "10"]
1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 *`;
    DOM.pgnTextArea.value = samplePgn;
    DOM.pgnNameInput.value = "Ruy Lopez Main Line";
}
export function createSubRepertoireAndTrain(currentFen) {
    let positionFound = false;
    let nodeForCurrentPosition = null;
    const currentShortFen = currentFen.split(' ')[0];
    const cachedCurrentFen = state.fenLookupCache.get(currentShortFen);
    if (cachedCurrentFen && state.repertoire.fenMap[cachedCurrentFen]) {
        positionFound = true;
        nodeForCurrentPosition = state.repertoire.fenMap[cachedCurrentFen];
    } else {
        for (const fen in state.repertoire.fenMap) {
            if (fen.split(' ')[0] === currentShortFen) {
                positionFound = true;
                nodeForCurrentPosition = state.repertoire.fenMap[fen];
                state.fenLookupCache.set(currentShortFen, fen);
                break;
            }
        }
    }
    if (!positionFound) {
        UI.showPlainError("This position is not in your repertoire. Please choose a position from your repertoire.");
        return;
    }
    const createSubRepertoire = () => {
        const subRepertoire = {
            name: state.repertoire.name + " (from position)",
            pgn: state.repertoire.pgn,
            startFen: currentFen,
            playerColor: state.repertoire.playerColor,
            fenMap: {},
            learnedNodes: new Set()
        };
        const buildSubTree = (node) => {
            if (!node || !node.children || node.children.length === 0) {
                return [];
            }
            return node.children.map(child => {
                const newChild = { ...child };
                if (child.children && child.children.length > 0) {
                    newChild.children = buildSubTree(child);
                } else {
                    newChild.children = [];
                }
                newChild.value = 0;
                newChild.updated = null;
                return newChild;
            });
        };
        subRepertoire.tree = buildSubTree(nodeForCurrentPosition);
        const buildFenMap = (nodes, startFen) => {
            const chess = getChessInstance(startFen);
            subRepertoire.fenMap[startFen] = { children: nodes, fen: startFen };
            const traverse = (branch, currentChess) => {
                const parentFen = currentChess.fen();
                for (const node of branch) {
                    const nextChess = getChessInstance(currentChess.fen());
                    const move = nextChess.move(node.san, { sloppy: true });
                    if (move) {
                        node.from = move.from;
                        node.to = move.to;
                        node.parentFen = parentFen;
                        subRepertoire.fenMap[nextChess.fen()] = node;
                        if (node.children.length > 0) {
                            traverse(node.children, nextChess);
                        }
                    }
                }
            };
            traverse(nodes, chess);
            releaseChessInstances();
        };
        buildFenMap(subRepertoire.tree, subRepertoire.startFen);
        for (const fen in subRepertoire.fenMap) {
            const node = subRepertoire.fenMap[fen];
            if (node.value >= LEARNED_THRESHOLD) {
                subRepertoire.learnedNodes.add(fen.split(' ')[0]);
            }
        }
        return subRepertoire;
    };
    const subRepertoire = createSubRepertoire();
    state.originalRepertoire = { ...state.repertoire };
    state.originalRepertoire.tree = JSON.parse(JSON.stringify(state.repertoire.tree));
    state.originalRepertoire.fenMap = { ...state.repertoire.fenMap };
    state.originalRepertoire.shortFenMap = { ...state.repertoire.shortFenMap };
    if (state.originalRepertoire.learnedNodes) {
        state.originalRepertoire.learnedNodes = new Set(state.repertoire.learnedNodes);
    }
    state.repertoire = subRepertoire;
    state.branches = createBranches(state.repertoire.tree, state.repertoire.startFen);
    UI.updatePgnViewer(state.repertoire.tree, state.repertoire.startFen);
    UI.updateProgressCounters();
    state.ground.set({
        movable: {
            free: false,
            color: state.playerColor === 'white' ? 'white' : 'black',
            dests: new Map(),
            events: { after: onUserMove }
        }
    });
    ACTIONS.setSubRepertoireLayout(true);
    state.currentBranchIndex = -1;
    startTrainingSession();
}
export function restoreOriginalRepertoire() {
    if (!state.originalRepertoire || !state.repertoire) return;
    for (const fen in state.repertoire.fenMap) {
        if (state.repertoire.fenMap.hasOwnProperty(fen) && state.originalRepertoire.fenMap[fen]) {
            const subNode = state.repertoire.fenMap[fen];
            const originalNode = state.originalRepertoire.fenMap[fen];
            if (subNode.updated && (!originalNode.updated || subNode.updated > originalNode.updated)) {
                originalNode.value = subNode.value;
                originalNode.updated = subNode.updated;
            }
        }
    }
    if (state.repertoire.learnedNodes) {
        state.repertoire.learnedNodes.forEach(shortFen => {
            state.originalRepertoire.learnedNodes.add(shortFen);
        });
    }
    state.repertoire = state.originalRepertoire;
    state.originalRepertoire = null;
    ACTIONS.setSubRepertoireLayout(false);
    state.branches = createBranches(state.repertoire.tree, state.repertoire.startFen);
    UI.updatePgnViewer(state.repertoire.tree, state.repertoire.startFen);
    UI.updateProgressCounters();
    state.game.load(state.repertoire.startFen);
    updateBoard();
    state.currentBranchIndex = -1;
    startTrainingSession();
}
