// Logic for session management and flow control

import { state } from './state.js';
import { LEARNED_THRESHOLD } from './constants.js';
import { getChessInstance, releaseChessInstances, toColor, findBranchForMove } from './utils.js';
import { updateBoard, showSquareMark, clearSquareMark } from './board.js';
import { showFeedback, forceClearFeedback, updateOpeningTreeProgress, updateProgressCounters, showCompletedFeedback } from './ui.js';
import { saveProgress } from './progress.js';
import { enterExploreMode } from './actions.js';
export function onBoardMove(oldFen, newFen) {
}
export function startTrainingSession() {
    if (!state.branches || state.branches.length === 0) {
        return;
    }
    selectNextBranch();
}
export function selectNextBranch() {
    clearSquareMark();
    const branchesWithMovesInProgress = state.branches.filter(b => !b.practiced && isBranchInProgress(b));
    if (branchesWithMovesInProgress.length > 0) {
        state.currentBranchIndex = state.branches.indexOf(branchesWithMovesInProgress[0]);
        resetToBranchStart(state.branches[state.currentBranchIndex]);
        return;
    }
    const branchesNotFullyLearned = state.branches.filter(b => !b.practiced && !isBranchFullyLearned(b));
    if (branchesNotFullyLearned.length > 0) {
        state.currentBranchIndex = state.branches.indexOf(branchesNotFullyLearned[0]);
        resetToBranchStart(state.branches[state.currentBranchIndex]);
    } else {
        showCompletedFeedback("All branches have been practiced! Resetting session.");
        state.branches.forEach(b => {
            b.practiced = false;
            b.progressIndex = 0;
        });
        saveProgress();
        if (state.branches.length > 0) {
            const branchesWithMovesInProgress = state.branches.filter(b => isBranchInProgress(b));
            if (branchesWithMovesInProgress.length > 0) {
                state.currentBranchIndex = state.branches.indexOf(branchesWithMovesInProgress[0]);
            } else {
                state.currentBranchIndex = 0;
            }
            resetToBranchStart(state.branches[state.currentBranchIndex]);
        } else {
            state.currentBranchIndex = -1;
        }
    }
}
export function isBranchInProgress(branch) {
    for (const move of branch.moves) {
        let moveNode = null;
        const shortFen = move.fen.split(' ')[0];
        const cachedFen = state.fenLookupCache.get(shortFen);
        if (cachedFen && state.repertoire.fenMap[cachedFen]) {
            moveNode = state.repertoire.fenMap[cachedFen];
        } else {
            for (const fen in state.repertoire.fenMap) {
                if (fen.split(' ')[0] === shortFen) {
                    moveNode = state.repertoire.fenMap[fen];
                    state.fenLookupCache.set(shortFen, fen);
                    break;
                }
            }
        }
        if (moveNode && moveNode.value > 0 && moveNode.value < LEARNED_THRESHOLD) {
            return true;
        }
    }
    return false;
}
export function isBranchFullyLearned(branch) {
    const playerTurn = state.playerColor === 'white' ? 'w' : 'b';
    const tempChess = getChessInstance();
    for (let i = 0; i < branch.moves.length; i++) {
        const move = branch.moves[i];
        if (i === 0) {
            tempChess.load(branch.trunkFEN);
        } else {
            tempChess.load(branch.moves[i - 1].fen);
        }
        if (tempChess.turn() === playerTurn) {
            let moveNode = null;
            const shortFen = move.fen.split(' ')[0];
            const cachedFen = state.fenLookupCache.get(shortFen);
            if (cachedFen && state.repertoire.fenMap[cachedFen]) {
                moveNode = state.repertoire.fenMap[cachedFen];
            } else {
                for (const fen in state.repertoire.fenMap) {
                    if (fen.split(' ')[0] === shortFen) {
                        moveNode = state.repertoire.fenMap[fen];
                        state.fenLookupCache.set(shortFen, fen);
                        break;
                    }
                }
            }
            if (!moveNode || moveNode.value < LEARNED_THRESHOLD) {
                releaseChessInstances();
                return false;
            }
        }
    }
    releaseChessInstances();
    return true;
}
export function resetToBranchStart(branch) {
    if (!branch) {
        return;
    }
    branch.progressIndex = 0;
    state.game.load(branch.trunkFEN);
    if (isBranchFullyLearned(branch)) {
        branch.practiced = true;
        saveProgress();
        setTimeout(selectNextBranch, 500);
        return;
    }
    if (toColor(state.game) !== state.playerColor) {
        makeOpponentMove();
    } else {
        updateBoard();
    }
}
export function makeOpponentMove() {
    clearSquareMark();
    if (state.currentBranchIndex === -1) return;
    const branch = state.branches[state.currentBranchIndex];
    if (!branch || branch.progressIndex >= branch.moves.length || toColor(state.game) === state.playerColor) {
        return;
    }
    if (isBranchFullyLearned(branch)) {
        branch.practiced = true;
        saveProgress();
        setTimeout(selectNextBranch, 500);
        return;
    }
    const moveSan = branch.moves[branch.progressIndex].san;
    const move = state.game.move(moveSan, { sloppy: true });
    if (move) {
        branch.progressIndex++;
        updateBoard();
        if (branch.progressIndex >= branch.moves.length) {
            branch.practiced = true;
            saveProgress();
            updateOpeningTreeProgress();
            setTimeout(selectNextBranch, 1000);
        } else if (toColor(state.game) !== state.playerColor) {
            setTimeout(makeOpponentMove, 500);
        }
    }
}
export function updateOpponentProgress(userMoveFen) {
    const shortFen = userMoveFen.split(' ')[0];
    let userMoveNode = null;
    let parentFen = null;
    const cachedUserFen = state.fenLookupCache.get(shortFen);
    if (cachedUserFen && state.repertoire.fenMap[cachedUserFen]) {
        userMoveNode = state.repertoire.fenMap[cachedUserFen];
        parentFen = userMoveNode.parentFen;
    } else {
        for (const fen in state.repertoire.fenMap) {
            if (fen.split(' ')[0] === shortFen) {
                userMoveNode = state.repertoire.fenMap[fen];
                parentFen = userMoveNode.parentFen;
                state.fenLookupCache.set(shortFen, fen);
                break;
            }
        }
    }
    if (!userMoveNode || !parentFen) return;
    let opponentMoveNode = null;
    const parentShortFen = parentFen.split(' ')[0];
    const cachedParentFen = state.fenLookupCache.get(parentShortFen);
    if (cachedParentFen && state.repertoire.fenMap[cachedParentFen]) {
        opponentMoveNode = state.repertoire.fenMap[cachedParentFen];
    } else {
        for (const fen in state.repertoire.fenMap) {
            if (fen.split(' ')[0] === parentShortFen) {
                opponentMoveNode = state.repertoire.fenMap[fen];
                state.fenLookupCache.set(parentShortFen, fen);
                break;
            }
        }
    }
    if (!opponentMoveNode || !opponentMoveNode.children || opponentMoveNode.children.length === 0) {
        return;
    }
    const allRepliesLearned = opponentMoveNode.children.every(replyNode => {
        return (replyNode.value || 0) >= LEARNED_THRESHOLD;
    });
    if (allRepliesLearned) {
        opponentMoveNode.value = LEARNED_THRESHOLD;
        opponentMoveNode.updated = Date.now();
    }
}
export function handleIncorrectMove(expectedMoveSan, destSquare, parentFen) {
    showSquareMark(destSquare, false);
    showFeedback(false, expectedMoveSan);
    const shortParentFen = parentFen.split(' ')[0];
    let parentNode = null;
    const cachedParentFen = state.fenLookupCache.get(shortParentFen);
    if (cachedParentFen && state.repertoire.fenMap[cachedParentFen]) {
        parentNode = state.repertoire.fenMap[cachedParentFen];
    } else {
        for (const fen in state.repertoire.fenMap) {
            if (fen.split(' ')[0] === shortParentFen) {
                parentNode = state.repertoire.fenMap[fen];
                state.fenLookupCache.set(shortParentFen, fen);
                break;
            }
        }
    }
    if (parentNode && parentNode.children) {
        parentNode.children.forEach(childNode => {
            if (childNode.value > 0 && childNode.value < LEARNED_THRESHOLD) {
                childNode.value = 0;
                childNode.updated = Date.now();
            }
        });
        saveProgress();
        updateOpeningTreeProgress();
        updateProgressCounters();
    }
    state.game.undo();
    updateBoard();
}
export function onUserMove(orig, dest, captured) {
    if (state.isExploreMode) return;
    clearSquareMark();
    if (state.currentBranchIndex === -1 || toColor(state.game) !== state.playerColor) return;
    const branch = state.branches[state.currentBranchIndex];
    const preMoveFen = state.game.fen();
    const tempGame = getChessInstance(preMoveFen);
    const move = tempGame.move({ from: orig, to: dest, promotion: 'q' });
    if (move === null) {
        return;
    }
    const deviatedBranch = findBranchForMove(preMoveFen, move.san);
    if (deviatedBranch && deviatedBranch.id !== branch.id) {
        const newIndex = state.branches.findIndex(b => b.id === deviatedBranch.id);
        if (newIndex !== -1) {
            state.currentBranchIndex = newIndex;
            resetToBranchStart(state.branches[state.currentBranchIndex]);
        }
        return;
    }
    state.ground.move(orig, dest);
    state.game.move({ from: orig, to: dest, promotion: 'q' });
    forceClearFeedback();
    const expectedMoveSan = branch.moves[branch.progressIndex].san;
    if (move.san === expectedMoveSan) {
        showSquareMark(dest, true);
        const resultFen = state.game.fen();
        const shortFen = resultFen.split(' ')[0];
        let moveNode = null;
        const cachedFen = state.fenLookupCache.get(shortFen);
        if (cachedFen && state.repertoire.fenMap[cachedFen]) {
            moveNode = state.repertoire.fenMap[cachedFen];
        } else {
            for (const fen in state.repertoire.fenMap) {
                if (fen.split(' ')[0] === shortFen) {
                    moveNode = state.repertoire.fenMap[fen];
                    state.fenLookupCache.set(shortFen, fen);
                    break;
                }
            }
        }
        if (moveNode) {
            const oldValue = moveNode.value || 0;
            moveNode.value = Math.min(LEARNED_THRESHOLD, oldValue + 1);
            moveNode.updated = Date.now();
            if (moveNode.value === LEARNED_THRESHOLD && oldValue < LEARNED_THRESHOLD) {
                updateOpponentProgress(resultFen);
            }
            saveProgress();
            updateProgressCounters();
        }
        branch.progressIndex++;
        showFeedback(true);
        if (branch.progressIndex >= branch.moves.length) {
            branch.practiced = true;
            saveProgress();
            updateOpeningTreeProgress();
            setTimeout(selectNextBranch, 1000);
        } else {
            setTimeout(makeOpponentMove, 500);
        }
    } else {
        handleIncorrectMove(expectedMoveSan, dest, preMoveFen);
    }
}
