// Updates training-specific UI components like stats, feedback and headers

import { renderOpeningTree } from './openingTree.js';
import { addFenToTree } from './logic.js';
import * as ACTIONS from './actions.js';
import { LEARNED_THRESHOLD } from './constants.js';
import { state } from './state.js';
import * as DOM from './dom.js';
import { getChessInstance, releaseChessInstances, toColor } from './utils.js';
export function updateMoveHighlight() {
    const currentFen = state.game.fen();
    for (const [fen, moveEl] of state.moveElementsCache) {
        if (moveEl.classList.contains('active')) {
            moveEl.classList.remove('active');
        }
        if (fen === currentFen) {
            moveEl.classList.add('active');
        }
    }
}
export function updatePgnViewer(openingTree, startFen) {
    if (!openingTree || openingTree.length === 0) {
        DOM.pgnViewerElement.innerHTML = '<p>No repertoire loaded or tree is empty</p>';
        return;
    }
    addFenToTree(openingTree, getChessInstance(startFen));
    releaseChessInstances();
    DOM.pgnViewerElement.classList.remove('loading');
    DOM.pgnViewerElement.style.textAlign = 'left';
    renderOpeningTree(DOM.pgnViewerElement, openingTree, null, null, state.playerColor);
    state.moveElementsCache.clear();
    const moveElements = DOM.pgnViewerElement.querySelectorAll('.ot-move-text');
    moveElements.forEach(moveEl => {
        if (moveEl.dataset.fen) {
            state.moveElementsCache.set(moveEl.dataset.fen, moveEl);
        }
        moveEl.addEventListener('click', (e) => {
            const fen = e.target.dataset.fen;
            ACTIONS.onPgnMoveClick(fen);
        });
    });
}
export function forceClearFeedback() {
    DOM.feedbackContainerElement.innerHTML = '';
    DOM.feedbackContainerElement.style.display = 'none';
}
export function clearFeedback() {
    const hasIncorrectFeedback = DOM.feedbackContainerElement.querySelector('.incorrect');
    if (!hasIncorrectFeedback) {
        DOM.feedbackContainerElement.innerHTML = '';
        DOM.feedbackContainerElement.style.display = 'none';
    }
}
export function showFeedback(correct, expectedMoveSan = null) {
    if (state.isExploreMode) return;
    clearFeedback();
    const feedbackEl = document.createElement('div');
    feedbackEl.classList.add('feedback-item');
    if (correct) {
        feedbackEl.classList.add('correct');
        feedbackEl.innerHTML = `<span class="feedback-icon">✓</span> <strong>Correct!</strong>`;
    } else {
        feedbackEl.classList.add('incorrect');
        let message = `The correct move was <span class="blurred-move">${expectedMoveSan || "different"}</span>.`;
        feedbackEl.innerHTML = `<span class="feedback-icon">✗</span> <strong>Incorrect.</strong> ${message}`;
        setTimeout(() => {
            const blurredMove = feedbackEl.querySelector('.blurred-move');
            if (blurredMove) {
                blurredMove.addEventListener('click', function () { this.classList.add('revealed'); });
            }
        }, 0);
    }
    DOM.feedbackContainerElement.appendChild(feedbackEl);
    DOM.feedbackContainerElement.style.display = 'block';
}
export function showCompletedFeedback(message) {
    DOM.feedbackContainerElement.innerHTML = `
        <div class="feedback-item correct">
            <strong>Congratulations!</strong>
            <p>${message || 'All due repertoire lines have been reviewed.'}</p>
        </div>
    `;
    DOM.feedbackContainerElement.style.display = 'block';
}
export function showError(message) {
    showFeedback(false, message);
}
export function showPlainError(message) {
    clearFeedback();
    const feedbackEl = document.createElement('div');
    feedbackEl.classList.add('feedback-item');
    feedbackEl.innerHTML = `<strong>${message}</strong>`;
    DOM.feedbackContainerElement.appendChild(feedbackEl);
    DOM.feedbackContainerElement.style.display = 'block';
}
export function showHint() {
    if (state.currentBranchIndex === -1 || toColor(state.game) !== state.playerColor) return;
    const branch = state.branches[state.currentBranchIndex];
    if (branch.progressIndex >= branch.moves.length) return;
    const expectedMoveSan = branch.moves[branch.progressIndex].san;
    const tempGame = getChessInstance(state.game.fen());
    const move = tempGame.move(expectedMoveSan, { sloppy: true });
    if (move) {
        state.ground.setShapes([{ orig: move.from, brush: 'blue' }]);
        setTimeout(() => {
            if (state.ground.state.drawable.shapes.some(s => s.orig === move.from && !s.dest)) {
                state.ground.setShapes([]);
            }
        }, 2000);
    }
}
export function updateProgressStats() {
    if (!state.repertoire) return;
    const practicedCount = state.branches.filter(b => b.practiced).length;
    const totalBranches = state.branches.length;
    if (DOM.progressValueElement) {
        DOM.progressValueElement.textContent = `${practicedCount} / ${totalBranches} branches practiced`;
    }
    if (DOM.accuracyValueElement) {
        DOM.accuracyValueElement.textContent = '';
    }
}
export function updateProgressCounters() {
    if (!state.repertoire || !state.repertoire.fenMap) return;
    let notStarted = 0;
    let inProgress = 0;
    let learned = 0;
    const playerTurn = state.playerColor === 'white' ? 'w' : 'b';
    const tempChess = getChessInstance();
    const countedNodes = new Set();
    for (const fen in state.repertoire.fenMap) {
        const node = state.repertoire.fenMap[fen];
        if (!node.san || !node.parentFen || countedNodes.has(node)) continue;
        tempChess.load(node.parentFen);
        if (tempChess.turn() === playerTurn) {
            countedNodes.add(node);
            const value = node.value || 0;
            if (value === 0) {
                notStarted++;
            } else if (value < LEARNED_THRESHOLD) {
                inProgress++;
            } else {
                learned++;
            }
        }
    }
    releaseChessInstances();
    const countersElement = document.getElementById('progress-counters');
    if (countersElement) {
        const total = notStarted + inProgress + learned;
        let progressBarHtml = '<div class="progress-bar-container">';
        if (notStarted > 0) {
            const notStartedPercent = (notStarted / total) * 100;
            progressBarHtml += `<div class="progress-segment progress-segment-not-started" style="width: ${notStartedPercent}%">${notStarted}</div>`;
        }
        if (inProgress > 0) {
            const inProgressPercent = (inProgress / total) * 100;
            progressBarHtml += `<div class="progress-segment progress-segment-learning" style="width: ${inProgressPercent}%">${inProgress}</div>`;
        }
        if (learned > 0) {
            const learnedPercent = (learned / total) * 100;
            progressBarHtml += `<div class="progress-segment progress-segment-learned" style="width: ${learnedPercent}%">${learned}</div>`;
        }
        progressBarHtml += '</div>';
        countersElement.innerHTML = progressBarHtml;
    }
}
export function toggleSidebar() {
    DOM.sidebar.classList.toggle('active');
    DOM.mainContent.classList.toggle('sidebar-active');
    const isActive = DOM.sidebar.classList.contains('active');
    const boardWrapper = document.querySelector('.board-wrapper');
    if (boardWrapper) {
        if (isActive) {
            boardWrapper.classList.add('board-disabled');
        } else {
            boardWrapper.classList.remove('board-disabled');
        }
    }
    const isExpanded = DOM.sidebar.classList.contains('active');
    DOM.sidebarToggleBtn.setAttribute('aria-expanded', isExpanded);
    if (DOM.mobileMenuBtn) {
        DOM.mobileMenuBtn.setAttribute('aria-expanded', isExpanded);
    }
}
export function updateOpeningTreeProgress() {
    if (!state.repertoire || !state.moveElementsCache.size) return;
    if (!state.repertoire.learnedNodes) {
        state.repertoire.learnedNodes = new Set();
    }
    for (const [fen, moveEl] of state.moveElementsCache) {
        const shortFen = fen.split(' ')[0];
        let node = null;
        const cachedFen = state.fenLookupCache.get(shortFen);
        if (cachedFen && state.repertoire.fenMap[cachedFen]) {
            node = state.repertoire.fenMap[cachedFen];
        } else {
            for (const mapFen in state.repertoire.fenMap) {
                if (mapFen.split(' ')[0] === shortFen) {
                    node = state.repertoire.fenMap[mapFen];
                    state.fenLookupCache.set(shortFen, mapFen);
                    break;
                }
            }
        }
        moveEl.classList.remove('ot-move-not-started', 'ot-move-learning', 'ot-move-learned');
        let statusClass = 'ot-move-not-started';
        if (node) {
            const value = node.value || 0;
            if (value >= LEARNED_THRESHOLD) {
                state.repertoire.learnedNodes.add(shortFen);
            }
            if (state.repertoire.learnedNodes.has(shortFen)) {
                statusClass = 'ot-move-learned';
            } else if (value === 0) {
                statusClass = 'ot-move-not-started';
            } else if (value < LEARNED_THRESHOLD) {
                statusClass = 'ot-move-learning';
            } else {
                statusClass = 'ot-move-learned';
            }
        }
        moveEl.classList.add(statusClass);
    }
}
