// Entry point for the training page, loading repertoire data based on URL parameters

import { initTraining } from "/static/js/training/main.js?v=1.0.1";
import { initImportPgn } from "/static/js/importPgn.js";
import { initAuth } from "/static/js/auth.js";
async function checkAuthStatus() {
    try {
        const response = await fetch("/api/me");
        if (response.ok) {
            const userData = await response.json();
            window.currentUser = userData;
            return userData;
        } else {
            window.currentUser = { logged_in: false };
            return { logged_in: false };
        }
    } catch (error) {
        window.currentUser = { logged_in: false };
        return { logged_in: false };
    }
}
document.addEventListener("DOMContentLoaded", async () => {
    const urlParams = new URLSearchParams(window.location.search);
    const repertoireIdentifier = urlParams.get("repertoire");
    const repertoireTitle = document.getElementById("opening-name");
    const pgnContainer = document.getElementById("pgn-viewer");
    initAuth();
    initImportPgn();
    if (!repertoireIdentifier) {
        repertoireTitle.textContent = "";
        pgnContainer.innerHTML = "<p>Please select a repertoire to start training.</p>";
        initTraining();
        return;
    }
    let pgn = "";
    let repertoireName = "";
    let repertoireColor = "white";
    try {
        const userData = await checkAuthStatus();
        if (userData.logged_in) {
            const response = await fetch(`/api/repertoires/${repertoireIdentifier}`);
            if (!response.ok) {
                throw new Error(`Repertoire with ID ${repertoireIdentifier} not found or permission denied.`);
            }
            const repertoire = await response.json();
            pgn = repertoire.pgn_data;
            repertoireName = repertoire.name;
            repertoireColor = repertoire.color;
        } else {
            const savedRepertoires = JSON.parse(localStorage.getItem("saved_repertoires") || "[]");
            const repertoire = savedRepertoires.find((r) => r.name === repertoireIdentifier);
            if (repertoire) {
                pgn = repertoire.pgn;
                repertoireName = repertoire.name;
                repertoireColor = repertoire.color;
            } else {
                throw new Error("Repertoire not found in local storage. Please log in to access server repertoires.");
            }
        }
        if (pgn) {
            repertoireTitle.textContent = `${repertoireName}`;
            pgnContainer.innerHTML = "";
            initTraining(pgn, repertoireName, repertoireColor);
        } else {
            throw new Error("Could not load the PGN data for the selected repertoire.");
        }
    } catch (error) {
        repertoireTitle.textContent = "Training: Error";
        pgnContainer.innerHTML = `<p style="color: red;">${error.message}</p>`;
        initTraining();
    }
});
